const fs = require('fs').promises;
const path = require('path');
const readline = require('readline');

const rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout
});

const question = (query) => new Promise((resolve) => rl.question(query, resolve));

async function findPHPFiles(directory) {
    const files = await fs.readdir(directory);
    const phpFiles = [];

    for (const file of files) {
        const filePath = path.join(directory, file);
        const stats = await fs.stat(filePath);

        if (stats.isDirectory()) {
            const subDirFiles = await findPHPFiles(filePath);
            phpFiles.push(...subDirFiles);
        } else if (path.extname(file) === '.php') {
            phpFiles.push(filePath);
        }
    }

    return phpFiles;
}

async function updateFooterContent(filePath) {
    try {
        let content = await fs.readFile(filePath, 'utf8');
        
        const footerRegex = /<footer[^>]*>[\s\S]*?<\/footer>/i;
        const footerMatch = content.match(footerRegex);

        if (footerMatch) {
            let footerContent = footerMatch[0];
            let hasChanges = false;

            // Expressions régulières pour la détection
            const followUsBlockRegex = /<div[^>]*>[\s\S]*?Follow us[\s\S]*?<\/div>/i;
            const instagramRegex = /<[^>]*>[\s\S]*?Instagram[\s\S]*?<\/[^>]*>/i;
            const emptyDivRegex = /<div[^>]*>\s*<\/div>/g;
            const multipleSpacesRegex = />\s{2,}</g;

            // Mise à jour de l'année 2024 à 2025
            if (footerContent.includes('2024')) {
                footerContent = footerContent.replace(/2024/g, '2025');
                hasChanges = true;
                console.log(`✅ Année mise à jour dans: ${filePath}`);
            }

            // Recherche du bloc "Follow us" ou du texte Instagram
            const followUsMatch = footerContent.match(followUsBlockRegex);
            const instagramMatch = footerContent.match(instagramRegex);

            if (followUsMatch || instagramMatch) {
                console.log('\nDans le fichier:', filePath);
                
                if (followUsMatch) {
                    console.log('Bloc "Follow us" trouvé:');
                    console.log(followUsMatch[0]);
                }
                
                if (instagramMatch) {
                    console.log('Élément Instagram trouvé:');
                    console.log(instagramMatch[0]);
                }

                const answer = await question('Voulez-vous supprimer ces éléments ? (oui/non): ');

                if (answer.toLowerCase() === 'oui') {
                    // Suppression du bloc "Follow us" s'il existe
                    if (followUsMatch) {
                        footerContent = footerContent.replace(followUsBlockRegex, '');
                    }
                    
                    // Suppression des références Instagram si elles existent
                    if (instagramMatch) {
                        footerContent = footerContent.replace(instagramRegex, '');
                    }

                    // Nettoyage des divs vides et des espaces excessifs
                    footerContent = footerContent
                        .replace(emptyDivRegex, '') // Supprime les divs vides
                        .replace(multipleSpacesRegex, '><') // Réduit les espaces entre les balises
                        .replace(/\s+/g, ' ') // Réduit les espaces multiples en un seul espace
                        .replace(/>\s+</g, '><'); // Supprime les espaces entre les balises

                    hasChanges = true;
                    console.log('✅ Éléments supprimés et mise en page optimisée');
                } else {
                    console.log('ℹ️ Éléments conservés');
                }
            }

            if (hasChanges) {
                content = content.replace(footerRegex, footerContent);
                await fs.writeFile(filePath, content, 'utf8');
                return true;
            }

            return false;
        } else {
            console.log(`⚠️ Pas de footer trouvé: ${filePath}`);
            return false;
        }
    } catch (error) {
        console.error(`❌ Erreur lors du traitement de ${filePath}:`, error);
        return false;
    }
}

async function main() {
    try {
        const startDir = process.argv[2] || './';
        
        console.log('🔍 Recherche des fichiers PHP...');
        const phpFiles = await findPHPFiles(startDir);
        
        if (phpFiles.length === 0) {
            console.log('❌ Aucun fichier PHP trouvé.');
            rl.close();
            return;
        }

        console.log(`📁 ${phpFiles.length} fichiers PHP trouvés. Début de la mise à jour...`);
        
        let updatedCount = 0;
        for (const file of phpFiles) {
            const wasUpdated = await updateFooterContent(file);
            if (wasUpdated) updatedCount++;
        }

        console.log('\n📊 Résumé:');
        console.log(`Total de fichiers traités: ${phpFiles.length}`);
        console.log(`Fichiers mis à jour: ${updatedCount}`);
        
        rl.close();
    } catch (error) {
        console.error('❌ Erreur générale:', error);
        rl.close();
        process.exit(1);
    }
}

main();