document.addEventListener('DOMContentLoaded', function() {
    // Elements
    const hamburger = document.getElementById('hamburger');
    const mobileMenu = document.querySelector('.mobile-menu');
    const mobileOverlay = document.querySelector('.mobile-overlay');
    const backButtons = document.querySelectorAll('.back-button');
    const menuItems = document.querySelectorAll('.mobile-menu-item');
    const header = document.querySelector('.header');
    const submenus = document.querySelectorAll('.mobile-submenu-panel');
    const desktopMenuItems = document.querySelectorAll('.desktop-menu-item');
    const desktopOverlays = document.querySelectorAll('.desktop-overlay');
    const overlayCloseButtons = document.querySelectorAll('.overlay-close');
    
    // Variables
    let isTransitioning = false;
    let lastScrollTop = 0;
    let scrollTimeout;
    let activeOverlay = null;

    // Desktop menu overlays
    desktopMenuItems.forEach(item => {
        item.addEventListener('click', (e) => {
            e.stopPropagation();
            const menuText = item.textContent.trim().split('\n')[0];
            const overlay = document.getElementById(`${menuText.toLowerCase().replace(/\s+/g, '-')}-overlay`);
            
            if (overlay) {
                // Close any active overlay
                if (activeOverlay && activeOverlay !== overlay) {
                    activeOverlay.classList.remove('active');
                }
                
                // Toggle current overlay
                if (overlay.classList.contains('active')) {
                    overlay.classList.remove('active');
                    activeOverlay = null;
                    document.body.style.overflow = '';
                } else {
                    overlay.classList.add('active');
                    activeOverlay = overlay;
                    document.body.style.overflow = 'hidden';
                }
            }
        });
    });

    // Close overlay when clicking outside
    document.addEventListener('click', (e) => {
        if (activeOverlay && !e.target.closest('.desktop-menu-item') && 
            !e.target.closest('.overlay-container')) {
            activeOverlay.classList.remove('active');
            activeOverlay = null;
            document.body.style.overflow = '';
        }
    });

    // Close overlay with close button
    overlayCloseButtons.forEach(button => {
        button.addEventListener('click', () => {
            const overlay = button.closest('.desktop-overlay');
            if (overlay) {
                overlay.classList.remove('active');
                activeOverlay = null;
                document.body.style.overflow = '';
            }
        });
    });

    // Mobile menu functions
    function openMenu() {
        if (isTransitioning) return;
        document.body.style.overflow = 'hidden';
        mobileOverlay.classList.add('active');
        mobileMenu.classList.add('active');
        hamburger.classList.add('active');
    }

    function closeMenu() {
        if (isTransitioning) return;
        document.body.style.overflow = '';
        mobileOverlay.classList.remove('active');
        mobileMenu.classList.remove('active');
        hamburger.classList.remove('active');
        submenus.forEach(submenu => {
            submenu.classList.remove('active');
        });
    }

    // Hamburger menu toggle
    hamburger.addEventListener('click', () => {
        mobileMenu.classList.contains('active') ? closeMenu() : openMenu();
    });

    // Back buttons in mobile submenus
    backButtons.forEach(button => {
        button.addEventListener('click', () => {
            const submenuPanel = button.closest('.mobile-submenu-panel');
            if (submenuPanel) {
                submenuPanel.classList.remove('active');
            }
        });
    });

    // Mobile overlay click to close
    mobileOverlay.addEventListener('click', closeMenu);

    // Mobile menu items click to open submenu
    menuItems.forEach(item => {
        item.addEventListener('click', () => {
            if (isTransitioning) return;
            isTransitioning = true;

            const submenuId = item.getAttribute('data-submenu') + '-submenu';
            const submenu = document.getElementById(submenuId);

            if (submenu) {
                submenu.classList.add('active');
            }

            setTimeout(() => {
                isTransitioning = false;
            }, 300);
        });
    });

    // Scroll handling for header
    window.addEventListener('scroll', () => {
        clearTimeout(scrollTimeout);
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        
        if (header.classList.contains('header-transparent')) {
            if (scrollTop > 0) {
                header.classList.add('scrolled');
            } else {
                header.classList.remove('scrolled');
            }
        }

        if (!mobileMenu.classList.contains('active') && Math.abs(scrollTop - lastScrollTop) > 50) {
            header.style.transform = scrollTop > lastScrollTop && scrollTop > header.offsetHeight ? 
                'translateY(-100%)' : 'translateY(0)';
            lastScrollTop = scrollTop;
        }

        scrollTimeout = setTimeout(() => {
            header.style.transform = 'translateY(0)';
        }, 1000);
    });

    // Touch gestures for mobile menu
    let touchStartY = 0;
    mobileMenu.addEventListener('touchstart', (e) => {
        touchStartY = e.touches[0].clientY;
    }, { passive: true });

    mobileMenu.addEventListener('touchmove', (e) => {
        const touchEndY = e.touches[0].clientY;
        const deltaY = touchEndY - touchStartY;

        if (deltaY > 70) {
            closeMenu();
            touchStartY = 0;
        }
    }, { passive: true });

    // ESC key to close overlays
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
            if (activeOverlay) {
                activeOverlay.classList.remove('active');
                activeOverlay = null;
                document.body.style.overflow = '';
            } else if (mobileMenu.classList.contains('active')) {
                closeMenu();
            }
        }
    });

    // Story cards click handling
    const storyCards = document.querySelectorAll('.story-card');
    storyCards.forEach(card => {
        card.addEventListener('click', () => {
            const link = card.querySelector('a');
            if (link) {
                link.click();
            }
        });
    });
});