<!DOCTYPE html>
<html lang="en-GB">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Store - Imators</title>
    <link href="https://cdn.imators.com/logo.png" rel="icon" type="image/png" />
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/tailwindcss@2.1.2/dist/tailwind.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        @media (max-width: 768px) {
            .apps-scroll {
                padding: 0 16px;
            }
            
            .category-header {
                padding: 40px 16px 20px;
            }
            
            .scroll-arrow {
                display: none;
            }
            
            .search-bar {
                bottom: 16px;
                width: calc(100vw - 32px);
            }
            
            .app-detail-content {
                padding: 40px 16px;
            }
            
            .app-detail-close {
                top: 16px;
                right: 16px;
            }
            
            .app-detail-header {
                flex-direction: column;
                gap: 16px;
            }
            
            .app-detail-specs {
                grid-template-columns: 1fr;
            }
            
            .app-detail-actions {
                flex-direction: column;
            }
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: #000000;
            color: #ffffff;
            line-height: 1.4;
            overflow-x: hidden;
        }
        
        header {
            background-color: rgba(0, 0, 0, 0.8);
        }
        
        .hero-section {
            padding: 80px 0 60px;
            text-align: left;
            border-bottom: 1px solid #333;
            max-width: 1200px;
            margin: 0 auto;
            padding-left: 32px;
            padding-right: 32px;
        }
        
        .hero-title {
            font-size: 48px;
            font-weight: 300;
            margin-bottom: 16px;
            letter-spacing: -0.02em;
        }
        
        .hero-subtitle {
            font-size: 16px;
            color: #999;
            font-weight: 300;
            max-width: 600px;
        }
        
        .category-section {
            margin-bottom: 60px;
            max-width: 1200px;
            margin-left: auto;
            margin-right: auto;
        }
        
        .category-header {
            padding: 40px 32px 20px;
            border-bottom: 1px solid #333;
        }
        
        .category-title {
            font-size: 24px;
            font-weight: 300;
            color: #fff;
        }
        
        .category-count {
            font-size: 14px;
            color: #666;
            font-weight: 300;
        }
        
        .apps-container {
            position: relative;
            overflow: hidden;
        }
        
        .apps-scroll {
            display: flex;
            gap: 1px;
            overflow-x: auto;
            scroll-behavior: smooth;
            scrollbar-width: none;
            -ms-overflow-style: none;
            padding: 0 32px;
        }
        
        .apps-scroll::-webkit-scrollbar {
            display: none;
        }
        
        .app-item {
            background: #000;
            min-width: 280px;
            width: 280px;
            padding: 32px;
            border-right: 1px solid #333;
            transition: background 0.2s ease;
            cursor: pointer;
            position: relative;
            display: flex;
            flex-direction: column;
        }
        
        .app-item:hover {
            background: #111;
        }
        
        .app-icon {
            width: 48px;
            height: 48px;
            border-radius: 8px;
            margin-bottom: 20px;
            overflow: hidden;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .app-icon img {
            width: 100%;
            height: 100%;
            object-fit: contain;
        }
        
        .app-name {
            font-size: 16px;
            font-weight: 400;
            color: #fff;
            margin-bottom: 8px;
            line-height: 1.3;
        }
        
        .app-description {
            font-size: 13px;
            color: #999;
            font-weight: 300;
            margin-bottom: 16px;
            line-height: 1.4;
            height: 34px;
            overflow: hidden;
            flex-grow: 1;
        }
        
        .app-meta {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .app-price {
            font-size: 14px;
            color: #fff;
            font-weight: 400;
        }
        
        .app-downloads {
            font-size: 11px;
            color: #666;
            font-weight: 300;
        }
        
        .imators-badge {
            position: absolute;
            top: 16px;
            right: 16px;
            width: 8px;
            height: 8px;
            background: #fff;
            border-radius: 50%;
        }
        
        .scroll-arrow {
            position: absolute;
            top: 50%;
            transform: translateY(-50%);
            width: 48px;
            height: 48px;
            background: rgba(0, 0, 0, 0.8);
            backdrop-filter: blur(10px);
            border: 1px solid #333;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            z-index: 10;
            transition: all 0.3s ease;
            opacity: 0;
            pointer-events: none;
        }
        
        .scroll-arrow.visible {
            opacity: 1;
            pointer-events: auto;
        }
        
        .scroll-arrow:hover {
            background: rgba(0, 0, 0, 0.9);
            border-color: #666;
        }
        
        .scroll-arrow.left {
            left: 16px;
        }
        
        .scroll-arrow.right {
            right: 16px;
        }
        
        .scroll-arrow svg {
            width: 20px;
            height: 20px;
            fill: #fff;
        }
        
        .search-bar {
            position: fixed;
            bottom: 32px;
            left: 50%;
            transform: translateX(-50%);
            z-index: 100;
            background: rgba(0, 0, 0, 0.9);
            backdrop-filter: blur(20px);
            border: 1px solid #333;
            border-radius: 24px;
            padding: 12px 24px;
            width: 400px;
            max-width: calc(100vw - 64px);
        }
        
        .search-input {
            background: none;
            border: none;
            outline: none;
            color: #fff;
            font-size: 14px;
            font-weight: 300;
            width: 100%;
            font-family: inherit;
        }
        
        .search-input::placeholder {
            color: #666;
        }
        
        .separator {
            height: 1px;
            background: #333;
            margin: 0;
        }
        
        a.text-link {
            color: #fff;
            text-decoration: none;
            border-bottom: 1px solid transparent;
            transition: border-color 0.2s ease;
        }
        
        a.text-link:hover {
            border-bottom-color: #fff;
        }
        
        .blur-overlay {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            backdrop-filter: blur(2px);
            opacity: 0;
            transition: opacity 0.3s ease;
            pointer-events: none;
            z-index: 5;
        }
        
        .blur-overlay.active {
            opacity: 1;
        }
        
        .app-detail-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.95);
            backdrop-filter: blur(20px);
            z-index: 1000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
            overflow-y: auto;
        }
        
        .app-detail-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .app-detail-content {
            max-width: 900px;
            margin: 0 auto;
            padding: 60px 32px;
            position: relative;
        }
        
        .app-detail-close {
            position: absolute;
            top: 32px;
            right: 32px;
            width: 40px;
            height: 40px;
            background: none;
            border: 1px solid #333;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            transition: all 0.2s ease;
            color: #fff;
        }
        
        .app-detail-close:hover {
            border-color: #666;
            background: rgba(255, 255, 255, 0.05);
        }
        
        .app-detail-header {
            display: flex;
            gap: 24px;
            margin-bottom: 40px;
            align-items: flex-start;
        }
        
        .app-detail-icon {
            width: 80px;
            height: 80px;
            border-radius: 16px;
            overflow: hidden;
            flex-shrink: 0;
        }
        
        .app-detail-icon img {
            width: 100%;
            height: 100%;
            object-fit: contain;
        }
        
        .app-detail-info h1 {
            font-size: 32px;
            font-weight: 300;
            margin-bottom: 8px;
            color: #fff;
        }
        
        .app-detail-meta {
            display: flex;
            gap: 24px;
            margin-bottom: 16px;
            flex-wrap: wrap;
        }
        
        .app-detail-meta span {
            font-size: 14px;
            color: #999;
            font-weight: 300;
        }
        
        .app-detail-price {
            font-size: 24px;
            font-weight: 300;
            color: #fff;
            margin-bottom: 16px;
        }
        
        .app-detail-description {
            font-size: 16px;
            color: #ccc;
            font-weight: 300;
            line-height: 1.6;
            margin-bottom: 32px;
        }
        
        .app-detail-actions {
            display: flex;
            gap: 16px;
            margin-bottom: 40px;
        }
        
        .app-detail-btn {
            padding: 12px 32px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 400;
            cursor: pointer;
            transition: all 0.2s ease;
            border: none;
            font-family: inherit;
        }
        
        .app-detail-btn.primary {
            background: #fff;
            color: #000;
        }
        
        .app-detail-btn.primary:hover {
            background: #f0f0f0;
        }
        
        .app-detail-btn.secondary {
            background: none;
            color: #fff;
            border: 1px solid #333;
        }
        
        .app-detail-btn.secondary:hover {
            border-color: #666;
            background: rgba(255, 255, 255, 0.05);
        }
        
        .app-detail-section {
            margin-bottom: 40px;
        }
        
        .app-detail-section h2 {
            font-size: 20px;
            font-weight: 300;
            color: #fff;
            margin-bottom: 16px;
        }
        
        .app-detail-section p {
            font-size: 14px;
            color: #999;
            font-weight: 300;
            line-height: 1.6;
        }
        
        .app-detail-specs {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 24px;
        }
        
        .app-detail-spec {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid #333;
        }
        
        .app-detail-spec:last-child {
            border-bottom: none;
        }
        
        .app-detail-spec span:first-child {
            color: #999;
            font-weight: 300;
        }
        
        .app-detail-spec span:last-child {
            color: #fff;
            font-weight: 300;
        }
        
        .download-modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.9);
            backdrop-filter: blur(20px);
            z-index: 1100;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .download-modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .download-modal {
            background: #111;
            border: 1px solid #333;
            border-radius: 8px;
            padding: 32px;
            max-width: 500px;
            width: calc(100vw - 64px);
            text-align: center;
        }
        
        .download-modal h3 {
            font-size: 20px;
            font-weight: 300;
            color: #fff;
            margin-bottom: 16px;
        }
        
        .download-modal p {
            font-size: 14px;
            color: #ccc;
            font-weight: 300;
            line-height: 1.6;
            margin-bottom: 24px;
        }
        
        .download-modal-btn {
            background: #fff;
            color: #000;
            border: none;
            padding: 12px 32px;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 400;
            cursor: pointer;
            transition: all 0.2s ease;
            font-family: inherit;
        }
        
        .download-modal-btn:hover {
            background: #f0f0f0;
        }
        
        .download-progress {
            width: 100%;
            height: 4px;
            background: #333;
            border-radius: 2px;
            overflow: hidden;
            margin: 8px 0;
        }
        
        .download-progress-bar {
            height: 100%;
            background: #fff;
            width: 0%;
            transition: width 0.1s ease;
            border-radius: 2px;
        }
        
        .payment-notice {
            background: rgba(255, 255, 255, 0.05);
            border: 1px solid #333;
            border-radius: 6px;
            padding: 16px;
            margin-bottom: 24px;
            text-align: left;
        }
        
        .payment-notice h4 {
            font-size: 14px;
            font-weight: 400;
            color: #fff;
            margin-bottom: 8px;
        }
        
        .payment-notice p {
            font-size: 13px;
            color: #999;
            font-weight: 300;
            line-height: 1.5;
            margin: 0;
        }

        .imators-link {
            font-size: 14px;
            color: #999;
            font-weight: 300;
            margin-bottom: 16px;
        }

        .imators-link a {
            color: #fff;
            text-decoration: none;
            border-bottom: 1px solid transparent;
            transition: border-color 0.2s ease;
        }

        .imators-link a:hover {
            border-bottom-color: #fff;
        }

        .share-section {
            margin-top: 32px;
            padding-top: 32px;
            border-top: 1px solid #333;
        }

        .share-btn {
            padding: 12px 32px;
            background: none;
            color: #fff;
            border: 1px solid #333;
            border-radius: 6px;
            font-size: 14px;
            font-weight: 400;
            cursor: pointer;
            transition: all 0.2s ease;
            font-family: inherit;
        }

        .share-btn:hover {
            border-color: #666;
            background: rgba(255, 255, 255, 0.05);
        }

        .share-success {
            font-size: 12px;
            color: #0f0;
            margin-top: 8px;
        }
    </style>
</head>
<body>
    <div class="hero-section">
        <h1 class="hero-title flex items-center gap-3">
            <img src="https://cdn.imators.com/logo.png" alt="Logo" class="h-8 w-8 object-contain">
            Imators Store
        </h1>
        <p class="hero-subtitle">Applications and digital products designed for productivity and creativity.</p>
    </div>

    <main id="main-content">
        <div id="categories-container">
            
        </div>
    </main>

    <div class="search-bar">
        <input type="text" class="search-input" placeholder="Search applications..." id="search-input">
    </div>

    <div class="app-detail-overlay" id="app-detail-overlay">
        <div class="app-detail-content">
            <button class="app-detail-close" onclick="closeAppDetails()">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                    <path d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z"/>
                </svg>
            </button>
            <div id="app-detail-body">
                
            </div>
        </div>
    </div>

    <div class="download-modal-overlay" id="download-modal-overlay">
        <div class="download-modal">
            <div id="download-modal-content">
                
            </div>
        </div>
    </div>

    <div class="separator"></div>

    <footer style="padding: 60px 0 120px;">
        <div style="max-width: 1200px; margin: 0 auto; padding: 0 32px; text-align: left;">
            <div style="margin-bottom: 32px;">
                <h3 style="font-size: 18px; font-weight: 300; margin-bottom: 16px;">Support</h3>
                <div style="display: flex; gap: 32px; font-size: 14px;">
                    <a href="https://imators.com/support" class="text-link">Help Centre</a>
                    <a href="https://imators.com/contact-us" class="text-link">Contact</a>
                    <a href="https://developer.imators.com" class="text-link" target="_blank">Publish my application</a>
                </div>
            </div>
            <div style="font-size: 12px; color: #666;">
                <p>&copy; 2025 Imators. All rights reserved.</p>
            </div>
        </div>
    </footer>

    <script>
        let allApps = [];
        let searchTimeout;
        let arrowTimeout = {};

        const categoryNames = {
            'made_by_imators': 'Made by Imators',
            'productivity': 'Productivity',
            'development': 'Development', 
            'security': 'Security',
            'utilities': 'Utilities',
            'design': 'Design',
            'business': 'Business'
        };

        async function loadApps() {
            try {
                const response = await fetch('get_apps.php');
                const data = await response.json();
                allApps = data;
                renderCategories();
                handleHashChange();
            } catch (error) {
                console.error('Error loading apps:', error);
            }
        }

        function groupAppsByCategory(apps) {
            const grouped = {};
            apps.forEach(app => {
                if (!grouped[app.category]) {
                    grouped[app.category] = [];
                }
                grouped[app.category].push(app);
            });
            return grouped;
        }

        function renderCategories() {
            const container = document.getElementById('categories-container');
            const grouped = groupAppsByCategory(allApps);
            
            const categoryOrder = ['made_by_imators', 'productivity', 'development', 'security', 'utilities', 'design', 'business'];
            
            container.innerHTML = categoryOrder.map(categoryKey => {
                const apps = grouped[categoryKey] || [];
                if (apps.length === 0) return '';
                
                return `
                    <div class="category-section">
                        <div class="category-header">
                            <div class="category-title">${categoryNames[categoryKey]}</div>
                            <div class="category-count">${apps.length} application${apps.length > 1 ? 's' : ''}</div>
                        </div>
                        <div class="apps-container" data-category="${categoryKey}">
                            <div class="blur-overlay"></div>
                            <div class="scroll-arrow left" onclick="scrollCategory('${categoryKey}', 'left')">
                                <svg viewBox="0 0 24 24"><path d="M15.41 16.59L10.83 12l4.58-4.59L14 6l-6 6 6 6z"/></svg>
                            </div>
                            <div class="scroll-arrow right" onclick="scrollCategory('${categoryKey}', 'right')">
                                <svg viewBox="0 0 24 24"><path d="M8.59 16.59L13.17 12 8.59 7.41 10 6l6 6-6 6z"/></svg>
                            </div>
                            <div class="apps-scroll" id="scroll-${categoryKey}">
                                ${apps.map(app => createAppHTML(app)).join('')}
                            </div>
                        </div>
                    </div>
                `;
            }).join('');
            
            setupScrollListeners();
        }

        function createAppHTML(app) {
            const isImators = app.category === 'made_by_imators';
            const priceDisplay = app.price_type === 'free' ? 'Free' :
                               app.price_type === 'subscription' ? `£${app.price}/mo` :
                               `£${app.price}`;
            
            return `
                <div class="app-item" onclick="showAppDetails('${app.slug}')">
                    ${isImators ? '<div class="imators-badge"></div>' : ''}
                    <div class="app-icon">
                        <img src="${app.icon_url}" alt="${app.name}" onerror="this.style.display='none'">
                    </div>
                    <div class="app-name">${app.name}</div>
                    <div class="app-description">${app.description}</div>
                    <div class="app-meta">
                        <div class="app-price">${priceDisplay}</div>
                        ${app.downloads > 0 ? `<div class="app-downloads">${app.downloads.toLocaleString()}</div>` : ''}
                    </div>
                </div>
            `;
        }

        function setupScrollListeners() {
            document.querySelectorAll('.apps-container').forEach(container => {
                const category = container.dataset.category;
                const scrollElement = container.querySelector('.apps-scroll');
                const leftArrow = container.querySelector('.scroll-arrow.left');
                const rightArrow = container.querySelector('.scroll-arrow.right');
                const blurOverlay = container.querySelector('.blur-overlay');
                
                function updateArrows() {
                    const canScrollLeft = scrollElement.scrollLeft > 0;
                    const canScrollRight = scrollElement.scrollLeft < scrollElement.scrollWidth - scrollElement.clientWidth;
                    
                    if (canScrollLeft || canScrollRight) {
                        leftArrow.style.display = canScrollLeft ? 'flex' : 'none';
                        rightArrow.style.display = canScrollRight ? 'flex' : 'none';
                    } else {
                        leftArrow.style.display = 'none';
                        rightArrow.style.display = 'none';
                    }
                }
                
                function showArrows() {
                    if (window.innerWidth <= 768) return;
                    
                    updateArrows();
                    leftArrow.classList.add('visible');
                    rightArrow.classList.add('visible');
                    
                    clearTimeout(arrowTimeout[category]);
                    arrowTimeout[category] = setTimeout(() => {
                        hideArrows();
                    }, 2000);
                }
                
                function hideArrows() {
                    leftArrow.classList.remove('visible');
                    rightArrow.classList.remove('visible');
                    blurOverlay.classList.remove('active');
                }
                
                function showBlur() {
                    blurOverlay.classList.add('active');
                }
                
                function hideBlur() {
                    blurOverlay.classList.remove('active');
                }
                
                container.addEventListener('mouseenter', showArrows);
                container.addEventListener('mousemove', showArrows);
                container.addEventListener('mouseleave', hideArrows);
                
                leftArrow.addEventListener('mouseenter', showBlur);
                leftArrow.addEventListener('mouseleave', hideBlur);
                rightArrow.addEventListener('mouseenter', showBlur);
                rightArrow.addEventListener('mouseleave', hideBlur);
                
                scrollElement.addEventListener('scroll', updateArrows);
                
                let startX;
                let scrollLeft;
                
                scrollElement.addEventListener('touchstart', e => {
                    startX = e.touches[0].pageX - scrollElement.offsetLeft;
                    scrollLeft = scrollElement.scrollLeft;
                });
                
                scrollElement.addEventListener('touchmove', e => {
                    e.preventDefault();
                    const x = e.touches[0].pageX - scrollElement.offsetLeft;
                    const walk = (x - startX) * 2;
                    scrollElement.scrollLeft = scrollLeft - walk;
                });
                
                updateArrows();
            });
        }

        function scrollCategory(category, direction) {
            const scrollElement = document.getElementById(`scroll-${category}`);
            const scrollAmount = 300;
            
            if (direction === 'left') {
                scrollElement.scrollLeft -= scrollAmount;
            } else {
                scrollElement.scrollLeft += scrollAmount;
            }
        }

        function searchApps(query) {
            const container = document.getElementById('categories-container');
            
            if (!query.trim()) {
                renderCategories();
                return;
            }
            
            const filteredApps = allApps.filter(app => 
                app.name.toLowerCase().includes(query.toLowerCase()) ||
                app.description.toLowerCase().includes(query.toLowerCase())
            );
            
            if (filteredApps.length === 0) {
                container.innerHTML = `
                    <div class="category-section">
                        <div class="category-header">
                            <div class="category-title">No results found</div>
                            <div class="category-count">Try a different search term</div>
                        </div>
                    </div>
                `;
                return;
            }
            
            container.innerHTML = `
                <div class="category-section">
                    <div class="category-header">
                        <div class="category-title">Search Results</div>
                        <div class="category-count">${filteredApps.length} application${filteredApps.length > 1 ? 's' : ''} found</div>
                    </div>
                    <div class="apps-container" data-category="search">
                        <div class="blur-overlay"></div>
                        <div class="scroll-arrow left" onclick="scrollCategory('search', 'left')">
                            <svg viewBox="0 0 24 24"><path d="M15.41 16.59L10.83 12l4.58-4.59L14 6l-6 6 6 6z"/></svg>
                        </div>
                        <div class="scroll-arrow right" onclick="scrollCategory('search', 'right')">
                            <svg viewBox="0 0 24 24"><path d="M8.59 16.59L13.17 12 8.59 7.41 10 6l6 6-6 6z"/></svg>
                        </div>
                        <div class="apps-scroll" id="scroll-search">
                            ${filteredApps.map(app => createAppHTML(app)).join('')}
                        </div>
                    </div>
                </div>
            `;
            
            setupScrollListeners();
        }

        document.getElementById('search-input').addEventListener('input', (e) => {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(() => {
                searchApps(e.target.value);
            }, 300);
        });

        function showAppDetails(slug) {
            const app = allApps.find(a => a.slug === slug);
            if (!app) return;
            
            window.location.hash = `app/${slug}`;
            
            const isImators = app.category === 'made_by_imators';
            const priceDisplay = app.price_type === 'free' ? 'Free' :
                               app.price_type === 'subscription' ? `£${app.price} per month` :
                               `£${app.price}`;
            
            const actionButton = app.price_type === 'free' ? 'Download' :
                               app.price_type === 'subscription' ? 'Subscribe' : 'Buy Now';
            
            const appUrl = `${window.location.origin}${window.location.pathname}#app/${slug}`;
            
            document.getElementById('app-detail-body').innerHTML = `
                <div class="app-detail-header">
                    <div class="app-detail-icon">
                        <img src="${app.icon_url}" alt="${app.name}" onerror="this.style.display='none'">
                    </div>
                    <div class="app-detail-info">
                        <h1>${app.name} ${isImators ? '<span style="color: #666;">by Imators</span>' : ''}</h1>
                        <div class="app-detail-meta">
                            <span>${categoryNames[app.category]}</span>
                            ${app.version ? `<span>Version ${app.version}</span>` : ''}
                            ${app.downloads > 0 ? `<span>${app.downloads.toLocaleString()} downloads</span>` : ''}
                            ${app.rating > 0 ? `<span>★ ${app.rating}/5 (${app.rating_count} reviews)</span>` : ''}
                        </div>
                        <div class="app-detail-price">${priceDisplay}</div>
                    </div>
                </div>
                
                <div class="app-detail-description">
                    ${app.long_description || app.description}
                </div>

                
                ${app.price_type !== 'free' && !isImators ? `
                <div class="payment-notice">
                    <h4>Payment Information</h4>
                    <p>Payment for this application is processed directly by the developer through their own secure payment system. Imators acts as a marketplace and is not responsible for payment processing or transaction disputes.</p>
                </div>
                ` : ''}
                
                <div class="app-detail-actions">
                    <button class="app-detail-btn primary" onclick="handleAppAction('${app.slug}', '${app.price_type}', ${isImators})">${actionButton}</button>
                    <button class="app-detail-btn secondary">Try Demo</button>
                </div>
                
                <div class="app-detail-section">
                    <h2>Requirements</h2>
                    <p>${app.requirements || 'No specific requirements listed.'}</p>
                </div>
                
                <div class="app-detail-section">
                    <h2>Details</h2>
                    <div class="app-detail-specs">
                        <div>
                            <div class="app-detail-spec">
                                <span>Category</span>
                                <span>${categoryNames[app.category]}</span>
                            </div>
                            <div class="app-detail-spec">
                                <span>Price Type</span>
                                <span>${app.price_type === 'free' ? 'Free' : app.price_type === 'subscription' ? 'Subscription' : 'One-time purchase'}</span>
                            </div>
                            ${app.size_mb ? `<div class="app-detail-spec"><span>Size</span><span>${app.size_mb} MB</span></div>` : ''}
                        </div>
                        <div>
                            ${app.version ? `<div class="app-detail-spec"><span>Version</span><span>${app.version}</span></div>` : ''}
                            <div class="app-detail-spec">
                                <span>Updated</span>
                                <span>${app.updated_at ? new Date(app.updated_at).toLocaleDateString() : new Date(app.created_at).toLocaleDateString()}</span>
                            </div>
                            ${app.rating > 0 ? `<div class="app-detail-spec"><span>Rating</span><span>★ ${app.rating}/5</span></div>` : ''}
                        </div>
                    </div>
                </div>

                <div class="share-section">
                    <h2 style="font-size: 20px; font-weight: 300; color: #fff; margin-bottom: 16px;">Share</h2>
                    <button class="share-btn" onclick="copyShareLink('${appUrl}')">Copy Link</button>
                    <div id="share-success"></div>
                </div>
            `;
            
            document.getElementById('app-detail-overlay').classList.add('active');
            document.body.style.overflow = 'hidden';
        }
        
        function closeAppDetails() {
            document.getElementById('app-detail-overlay').classList.remove('active');
            document.body.style.overflow = 'auto';
            window.location.hash = '';
        }
        
        function handleHashChange() {
            const hash = window.location.hash.slice(1);
            
            if (hash.startsWith('app/')) {
                const slug = hash.replace('app/', '');
                showAppDetails(slug);
            } else {
                closeAppDetails();
            }
        }
        
        window.addEventListener('hashchange', handleHashChange);
        
        document.getElementById('app-detail-overlay').addEventListener('click', (e) => {
            if (e.target === e.currentTarget) {
                closeAppDetails();
            }
        });

        function handleAppAction(slug, priceType, isImators) {
            const app = allApps.find(a => a.slug === slug);
            if (!app) return;
            
            if (priceType === 'free') {
                downloadApp(app, isImators);
            } else {
                window.open(app.purchase_url || '#', '_blank');
            }
        }
        
        function downloadApp(app, isImators) {
            const button = event.target;
            const originalText = button.textContent;
            
            button.disabled = true;
            button.innerHTML = `
                <span>Downloading...</span>
                <div class="download-progress">
                    <div class="download-progress-bar" id="progress-bar"></div>
                </div>
            `;
            
            let progress = 0;
            const progressBar = document.getElementById('progress-bar');
            
            const interval = setInterval(() => {
                progress += Math.random() * 15 + 5;
                if (progress > 100) progress = 100;
                
                progressBar.style.width = progress + '%';
                
                if (progress >= 100) {
                    clearInterval(interval);
                    
                    setTimeout(() => {
                        if (app.download_url) {
                            const link = document.createElement('a');
                            link.href = app.download_url;
                            link.download = app.name + '.zip';
                            document.body.appendChild(link);
                            link.click();
                            document.body.removeChild(link);
                        }
                        
                        button.disabled = false;
                        button.textContent = originalText;
                        
                        if (!isImators) {
                            showDownloadModal();
                        }
                    }, 500);
                }
            }, 100);
        }
        
        function showDownloadModal() {
            document.getElementById('download-modal-content').innerHTML = `
                <h3>Download Complete</h3>
                <p><strong>Important Notice:</strong> We are not responsible if you encounter any issues with this product.</p>
                <p>Our team analyses the source code of each project from A to Z, but is not dedicated to it. Our customer support is not dedicated to this project as we did not develop it. Please refer to the GitHub documentation if necessary.</p>
                <button class="download-modal-btn" onclick="closeDownloadModal()">I Understand</button>
            `;
            
            document.getElementById('download-modal-overlay').classList.add('active');
        }
        
        function closeDownloadModal() {
            document.getElementById('download-modal-overlay').classList.remove('active');
        }
        
        document.getElementById('download-modal-overlay').addEventListener('click', (e) => {
            if (e.target === e.currentTarget) {
                closeDownloadModal();
            }
        });

        function copyShareLink(url) {
            navigator.clipboard.writeText(url).then(() => {
                const successDiv = document.getElementById('share-success');
                successDiv.innerHTML = '<div class="share-success">Link copied to clipboard!</div>';
                setTimeout(() => {
                    successDiv.innerHTML = '';
                }, 3000);
            }).catch(() => {
                alert('Failed to copy link');
            });
        }

        loadApps();
    </script>
</body>
</html>