<!DOCTYPE html>
<html lang="fr" class="h-full">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>TLCA - Base de Renseignement</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'media'
        }
    </script>
</head>
<body class="h-full bg-white dark:bg-black text-black dark:text-white transition-colors duration-300">
    <div class="min-h-screen p-8">
        <header class="text-center mb-16">
            <h1 class="text-4xl font-extrabold mb-4 tracking-tight">TLCA DATABASE SERVICES</h1>
            <div class="flex items-center justify-center space-x-2 opacity-60">
                <div class="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                <span class="text-sm">Connexion sécurisée active</span>
            </div>
        </header>

        <div class="max-w-7xl mx-auto">
            <div class="mb-8">
                <div class="flex flex-col sm:flex-row gap-4">
                    <input type="text" id="searchInput" placeholder="Rechercher par nom, prénom, localisation..." class="flex-1 px-6 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black focus:outline-none focus:ring-1 focus:ring-gray-400 dark:focus:ring-gray-600 transition-all">
                    <button onclick="openCreateModal()" class="px-8 py-3 bg-black dark:bg-white text-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors font-medium">Nouveau Profil</button>
                </div>
            </div>

            <div id="profilesList" class="space-y-6">
            </div>
        </div>
    </div>

    <div id="errorOverlay" class="fixed inset-0 bg-black bg-opacity-90 backdrop-blur-sm hidden items-center justify-center z-50">
        <div class="bg-red-600 text-white rounded-lg p-8 max-w-md mx-4 text-center">
            <h2 class="text-2xl font-bold mb-4">Accès Refusé</h2>
            <p class="mb-6">Votre adresse IP n'est pas autorisée à accéder à cette base de données sécurisée.</p>
            <p class="text-sm opacity-80">Code d'erreur: 403 - Unauthorized Access</p>
        </div>
    </div>

    <div id="createModal" class="fixed inset-0 bg-black bg-opacity-70 hidden items-center justify-center z-50">
        <div class="bg-white dark:bg-black rounded-lg p-8 w-full max-w-lg mx-4 border border-gray-200 dark:border-gray-800">
            <h3 class="text-2xl font-semibold mb-8 text-center">Créer un Profil</h3>
            <div class="space-y-6">
                <div class="grid grid-cols-2 gap-4">
                    <input type="text" id="createNom" placeholder="Nom" class="px-4 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black focus:outline-none focus:ring-1 focus:ring-gray-400">
                    <input type="text" id="createPrenom" placeholder="Prénom" class="px-4 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black focus:outline-none focus:ring-1 focus:ring-gray-400">
                </div>
                <input type="text" id="createLieu" placeholder="Lieu de dernière observation" class="w-full px-4 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black focus:outline-none focus:ring-1 focus:ring-gray-400">
                <textarea id="createInfos" placeholder="Informations additionnelles" class="w-full px-4 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black h-24 focus:outline-none focus:ring-1 focus:ring-gray-400"></textarea>
                <div class="flex items-center space-x-3">
                    <input type="checkbox" id="createRisque" class="w-4 h-4 rounded border-gray-300">
                    <label for="createRisque" class="font-medium">Profil à risque</label>
                </div>
                <textarea id="createRisqueDesc" placeholder="Description du risque (obligatoire si profil à risque)" class="w-full px-4 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black h-20 focus:outline-none focus:ring-1 focus:ring-gray-400 hidden"></textarea>
            </div>
            <div class="flex justify-center space-x-4 mt-8">
                <button onclick="closeCreateModal()" class="px-6 py-3 text-gray-600 dark:text-gray-400 hover:text-black dark:hover:text-white transition-colors">Annuler</button>
                <button onclick="createProfile()" class="px-8 py-3 bg-black dark:bg-white text-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors font-medium">Créer</button>
            </div>
        </div>
    </div>

    <div id="profileModal" class="fixed inset-0 bg-black bg-opacity-70 hidden items-center justify-center z-50">
        <div class="bg-white dark:bg-black rounded-lg p-8 w-full max-w-2xl mx-4 border border-gray-200 dark:border-gray-800 max-h-[90vh] overflow-y-auto">
            <div class="flex justify-between items-center mb-8">
                <h3 class="text-2xl font-semibold">Détails du Profil</h3>
                <button onclick="closeProfileModal()" class="text-2xl opacity-60 hover:opacity-100">&times;</button>
            </div>
            <div id="profileModalContent"></div>
        </div>
    </div>

    <div id="updateLocationModal" class="fixed inset-0 bg-black bg-opacity-70 hidden items-center justify-center z-50">
        <div class="bg-white dark:bg-black rounded-lg p-8 w-full max-w-md mx-4 border border-gray-200 dark:border-gray-800">
            <h3 class="text-xl font-semibold mb-6 text-center">Mettre à jour la localisation</h3>
            <div class="space-y-4">
                <input type="text" id="updateLieu" placeholder="Nouveau lieu d'observation" class="w-full px-4 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black focus:outline-none focus:ring-1 focus:ring-gray-400">
                <button onclick="getCurrentLocation()" class="w-full px-4 py-3 bg-gray-200 dark:bg-gray-800 text-black dark:text-white rounded-lg hover:bg-gray-300 dark:hover:bg-gray-700 transition-colors">Utiliser ma position actuelle</button>
            </div>
            <div class="flex justify-center space-x-4 mt-6">
                <button onclick="closeUpdateLocationModal()" class="px-6 py-3 text-gray-600 dark:text-gray-400 hover:text-black dark:hover:text-white transition-colors">Annuler</button>
                <button onclick="updateLocation()" class="px-8 py-3 bg-black dark:bg-white text-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors font-medium">Mettre à jour</button>
            </div>
        </div>
    </div>

    <script>
        let profiles = [];
        let selectedProfileId = null;
        let updateProfileId = null;

        function loadProfiles() {
            fetch('db.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({action: 'getProfiles'})
            })
            .then(response => {
                if (response.status === 403) {
                    document.getElementById('errorOverlay').classList.remove('hidden');
                    document.getElementById('errorOverlay').classList.add('flex');
                    return;
                }
                return response.json();
            })
            .then(data => {
                if (data && Array.isArray(data)) {
                    profiles = data;
                    displayProfiles(profiles);
                } else if (data && data.error) {
                    if (data.error.includes('Accès non autorisé')) {
                        document.getElementById('errorOverlay').classList.remove('hidden');
                        document.getElementById('errorOverlay').classList.add('flex');
                    } else {
                        console.error('Erreur:', data);
                        profiles = [];
                        displayProfiles([]);
                    }
                }
            })
            .catch(error => {
                console.error('Erreur de connexion:', error);
                profiles = [];
                displayProfiles([]);
            });
        }

        function displayProfiles(profilesToShow) {
            const container = document.getElementById('profilesList');
            container.innerHTML = '';
            
            if (!profilesToShow || profilesToShow.length === 0) {
                container.innerHTML = '<div class="text-center py-12 opacity-60"><p>Aucun profil trouvé</p></div>';
                return;
            }
            
            profilesToShow.forEach(profile => {
                const div = document.createElement('div');
                div.className = `bg-gray-50 dark:bg-gray-950 rounded-lg p-6 cursor-pointer hover:bg-gray-100 dark:hover:bg-gray-900 transition-colors border-l-4 ${profile.risque ? 'border-red-500' : 'border-gray-300 dark:border-gray-700'}`;
                div.onclick = () => showProfileDetails(profile.id);
                
                div.innerHTML = `
                    <div class="flex justify-between items-start">
                        <div class="flex-1">
                            <div class="flex items-center space-x-3 mb-3">
                                <h4 class="text-xl font-semibold">${profile.nom} ${profile.prenom}</h4>
                                ${profile.risque ? '<span class="bg-red-500 text-white text-xs px-2 py-1 rounded font-medium">RISQUE</span>' : ''}
                            </div>
                            <p class="text-sm opacity-70 mb-2">Dernière observation: ${profile.lieu}</p>
                            <p class="text-xs opacity-50">${new Date(profile.derniere_mise_a_jour).toLocaleString()}</p>
                        </div>
                        <div class="flex space-x-3 opacity-60">
                            <button onclick="openUpdateLocationModal(${profile.id}); event.stopPropagation();" class="hover:opacity-100 transition-opacity text-lg">📍</button>
                            <button onclick="deleteProfile(${profile.id}); event.stopPropagation();" class="hover:opacity-100 transition-opacity text-lg">🗑️</button>
                        </div>
                    </div>
                `;
                container.appendChild(div);
            });
        }

        function showProfileDetails(profileId) {
            const profile = profiles.find(p => p.id === profileId);
            selectedProfileId = profileId;
            
            const contentDiv = document.getElementById('profileModalContent');
            
            contentDiv.innerHTML = `
                <div class="space-y-6">
                    <div class="text-center pb-4 border-b border-gray-200 dark:border-gray-800">
                        <h4 class="text-2xl font-semibold mb-2">${profile.nom} ${profile.prenom}</h4>
                        ${profile.risque ? '<span class="inline-block bg-red-500 text-white text-sm px-3 py-1 rounded font-medium">PROFIL À RISQUE</span>' : ''}
                        <p class="text-sm opacity-60 mt-2">Dernière mise à jour: ${new Date(profile.derniere_mise_a_jour).toLocaleString()}</p>
                    </div>
                    
                    <div>
                        <h5 class="font-medium mb-3 opacity-80">Dernière localisation</h5>
                        <p class="text-lg mb-4 bg-gray-50 dark:bg-gray-950 p-3 rounded-lg">${profile.lieu}</p>
                    </div>
                    
                    <div>
                        <h5 class="font-medium mb-3 opacity-80">Informations générales</h5>
                        <div class="bg-gray-50 dark:bg-gray-950 p-4 rounded-lg">
                            <p class="text-sm leading-relaxed">${profile.informations || 'Aucune information disponible'}</p>
                        </div>
                    </div>
                    
                    ${profile.risque && profile.description_risque ? `
                    <div>
                        <h5 class="font-medium mb-3 text-red-500">Description du risque</h5>
                        <div class="bg-red-50 dark:bg-red-950/20 border border-red-200 dark:border-red-800 p-4 rounded-lg">
                            <p class="text-sm leading-relaxed">${profile.description_risque}</p>
                        </div>
                    </div>
                    ` : ''}
                    
                    <div>
                        <h5 class="font-medium mb-3 opacity-80">Historique des observations</h5>
                        <div class="space-y-2 max-h-40 overflow-y-auto bg-gray-50 dark:bg-gray-950 rounded-lg p-4">
                            ${profile.historique ? profile.historique.split('\n').map(h => `
                                <div class="text-sm bg-white dark:bg-gray-900 p-3 rounded border border-gray-200 dark:border-gray-800">
                                    ${h}
                                </div>
                            `).join('') : '<p class="text-sm opacity-60">Aucun historique disponible</p>'}
                        </div>
                    </div>
                    
                    <div class="flex justify-between pt-6 border-t border-gray-200 dark:border-gray-800">
                        <div class="flex space-x-3">
                            <button onclick="addInfoToProfile()" class="px-6 py-3 bg-gray-200 dark:bg-gray-800 text-black dark:text-white rounded-lg hover:bg-gray-300 dark:hover:bg-gray-700 transition-colors text-sm font-medium">Ajouter info</button>
                            <button onclick="openUpdateLocationModal(${profile.id})" class="px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors text-sm font-medium">Nouvelle observation</button>
                        </div>
                        <div class="flex space-x-3">
                            <button onclick="editProfile()" class="px-8 py-3 bg-black dark:bg-white text-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors font-medium">Modifier</button>
                            <button onclick="closeProfileModal()" class="px-6 py-3 text-gray-600 dark:text-gray-400 hover:text-black dark:hover:text-white transition-colors">Fermer</button>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('profileModal').classList.remove('hidden');
            document.getElementById('profileModal').classList.add('flex');
        }
        
        function editProfile() {
            const profile = profiles.find(p => p.id === selectedProfileId);
            const contentDiv = document.getElementById('profileModalContent');
            
            contentDiv.innerHTML = `
                <div class="space-y-6">
                    <div class="text-center pb-4 border-b border-gray-200 dark:border-gray-800">
                        <h4 class="text-xl font-semibold">Modifier le Profil</h4>
                    </div>
                    
                    <div class="grid grid-cols-2 gap-4">
                        <div>
                            <label class="block text-sm font-medium mb-2 opacity-80">Nom</label>
                            <input type="text" id="editNom" value="${profile.nom}" class="w-full px-4 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black focus:outline-none focus:ring-1 focus:ring-gray-400">
                        </div>
                        <div>
                            <label class="block text-sm font-medium mb-2 opacity-80">Prénom</label>
                            <input type="text" id="editPrenom" value="${profile.prenom}" class="w-full px-4 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black focus:outline-none focus:ring-1 focus:ring-gray-400">
                        </div>
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium mb-2 opacity-80">Dernière localisation</label>
                        <input type="text" id="editLieu" value="${profile.lieu}" class="w-full px-4 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black focus:outline-none focus:ring-1 focus:ring-gray-400">
                    </div>
                    
                    <div>
                        <label class="block text-sm font-medium mb-2 opacity-80">Informations générales</label>
                        <textarea id="editInfos" class="w-full px-4 py-3 border border-gray-200 dark:border-gray-800 rounded-lg bg-white dark:bg-black h-24 focus:outline-none focus:ring-1 focus:ring-gray-400">${profile.informations || ''}</textarea>
                    </div>
                    
                    <div class="flex items-center space-x-3">
                        <input type="checkbox" id="editRisque" ${profile.risque ? 'checked' : ''} class="w-4 h-4 rounded border-gray-300">
                        <label for="editRisque" class="font-medium">Profil à risque</label>
                    </div>
                    
                    <div id="editRisqueContainer" class="${profile.risque ? '' : 'hidden'}">
                        <label class="block text-sm font-medium mb-2 text-red-500">Description du risque</label>
                        <textarea id="editRisqueDesc" class="w-full px-4 py-3 border border-red-300 dark:border-red-700 rounded-lg bg-white dark:bg-black h-20 focus:outline-none focus:ring-1 focus:ring-red-400">${profile.description_risque || ''}</textarea>
                    </div>
                    
                    <div class="flex justify-center space-x-4 pt-6 border-t border-gray-200 dark:border-gray-800">
                        <button onclick="showProfileDetails(${selectedProfileId})" class="px-6 py-3 text-gray-600 dark:text-gray-400 hover:text-black dark:hover:text-white transition-colors">Annuler</button>
                        <button onclick="saveProfileChanges()" class="px-8 py-3 bg-black dark:bg-white text-white dark:text-black rounded-lg hover:bg-gray-800 dark:hover:bg-gray-200 transition-colors font-medium">Sauvegarder</button>
                    </div>
                </div>
            `;
            
            document.getElementById('editRisque').addEventListener('change', function() {
                const container = document.getElementById('editRisqueContainer');
                if (this.checked) {
                    container.classList.remove('hidden');
                } else {
                    container.classList.add('hidden');
                }
            });
        }
        
        function closeProfileModal() {
            document.getElementById('profileModal').classList.add('hidden');
            document.getElementById('profileModal').classList.remove('flex');
        }
        
        function saveProfileChanges() {
            const data = {
                action: 'updateProfile',
                id: selectedProfileId,
                nom: document.getElementById('editNom').value.trim(),
                prenom: document.getElementById('editPrenom').value.trim(),
                lieu: document.getElementById('editLieu').value.trim(),
                informations: document.getElementById('editInfos').value,
                risque: document.getElementById('editRisque').checked,
                description_risque: document.getElementById('editRisqueDesc').value
            };
            
            if (!data.nom || !data.prenom || !data.lieu) {
                alert('Nom, prénom et lieu sont obligatoires');
                return;
            }
            
            fetch('db.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify(data)
            })
            .then(response => response.json())
            .then(result => {
                if (result.success) {
                    loadProfiles();
                    showProfileDetails(selectedProfileId);
                } else {
                    alert('Erreur lors de la sauvegarde: ' + (result.error || 'Erreur inconnue'));
                }
            })
            .catch(error => {
                console.error('Erreur:', error);
                alert('Erreur de connexion');
            });
        }

        function searchProfiles() {
            const query = document.getElementById('searchInput').value.toLowerCase();
            if (!query) {
                displayProfiles(profiles);
                return;
            }
            
            const filtered = profiles.filter(profile => 
                profile.nom.toLowerCase().includes(query) ||
                profile.prenom.toLowerCase().includes(query) ||
                profile.lieu.toLowerCase().includes(query)
            );
            
            displayProfiles(filtered);
        }

        function openCreateModal() {
            document.getElementById('createModal').classList.remove('hidden');
            document.getElementById('createModal').classList.add('flex');
        }

        function closeCreateModal() {
            document.getElementById('createModal').classList.add('hidden');
            document.getElementById('createModal').classList.remove('flex');
            document.getElementById('createRisqueDesc').classList.add('hidden');
            document.getElementById('createNom').value = '';
            document.getElementById('createPrenom').value = '';
            document.getElementById('createLieu').value = '';
            document.getElementById('createInfos').value = '';
            document.getElementById('createRisque').checked = false;
            document.getElementById('createRisqueDesc').value = '';
        }

        function createProfile() {
            const nom = document.getElementById('createNom').value.trim();
            const prenom = document.getElementById('createPrenom').value.trim();
            const lieu = document.getElementById('createLieu').value.trim();
            
            if (!nom || !prenom || !lieu) {
                alert('Nom, prénom et lieu sont obligatoires');
                return;
            }
            
            const data = {
                action: 'createProfile',
                nom: nom,
                prenom: prenom,
                lieu: lieu,
                informations: document.getElementById('createInfos').value,
                risque: document.getElementById('createRisque').checked,
                description_risque: document.getElementById('createRisqueDesc').value
            };
            
            fetch('db.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify(data)
            })
            .then(response => response.json())
            .then(result => {
                if (result.success) {
                    closeCreateModal();
                    loadProfiles();
                } else {
                    alert('Erreur lors de la création: ' + (result.error || 'Erreur inconnue'));
                }
            })
            .catch(error => {
                console.error('Erreur:', error);
                alert('Erreur de connexion');
            });
        }

        function deleteProfile(profileId) {
            if (confirm('Êtes-vous certain de vouloir supprimer ce profil de manière définitive ?')) {
                fetch('db.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({action: 'deleteProfile', id: profileId})
                })
                .then(response => response.json())
                .then(result => {
                    loadProfiles();
                    if (selectedProfileId === profileId) {
                        closeProfileModal();
                    }
                });
            }
        }

        function openUpdateLocationModal(profileId) {
            updateProfileId = profileId;
            document.getElementById('updateLocationModal').classList.remove('hidden');
            document.getElementById('updateLocationModal').classList.add('flex');
        }

        function closeUpdateLocationModal() {
            document.getElementById('updateLocationModal').classList.add('hidden');
            document.getElementById('updateLocationModal').classList.remove('flex');
            document.getElementById('updateLieu').value = '';
        }

        function getCurrentLocation() {
            if (navigator.geolocation) {
                navigator.geolocation.getCurrentPosition(function(position) {
                    const lat = position.coords.latitude;
                    const lon = position.coords.longitude;
                    document.getElementById('updateLieu').value = `Position GPS: ${lat.toFixed(6)}, ${lon.toFixed(6)}`;
                });
            }
        }

        function updateLocation() {
            const newLocation = document.getElementById('updateLieu').value;
            
            fetch('db.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    action: 'updateLocation',
                    id: updateProfileId,
                    lieu: newLocation
                })
            })
            .then(response => response.json())
            .then(result => {
                closeUpdateLocationModal();
                loadProfiles();
                if (selectedProfileId === updateProfileId) {
                    showProfileDetails(updateProfileId);
                }
            });
        }

        function addInfoToProfile() {
            const newInfo = prompt('Ajouter des informations:');
            if (newInfo) {
                fetch('db.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        action: 'addInfo',
                        id: selectedProfileId,
                        info: newInfo
                    })
                })
                .then(response => response.json())
                .then(result => {
                    loadProfiles();
                    showProfileDetails(selectedProfileId);
                });
            }
        }

        document.getElementById('searchInput').addEventListener('input', searchProfiles);
        document.getElementById('searchInput').addEventListener('keyup', searchProfiles);

        document.getElementById('createRisque').addEventListener('change', function() {
            const riskDesc = document.getElementById('createRisqueDesc');
            if (this.checked) {
                riskDesc.classList.remove('hidden');
            } else {
                riskDesc.classList.add('hidden');
            }
        });

        loadProfiles();
        setInterval(loadProfiles, 5000);
    </script>
</body>
</html>