<?php
require_once 'db.php';
require_once 'stripe_payment.php';

class DeliveryProcessor {
    
    private $running = true;
    
    public function __construct() {
        register_shutdown_function([$this, 'shutdown']);
        pcntl_signal(SIGTERM, [$this, 'signalHandler']);
        pcntl_signal(SIGINT, [$this, 'signalHandler']);
    }
    
    public function run() {
        echo "[" . date('Y-m-d H:i:s') . "] Démarrage du processeur de livraison\n";
        
        while ($this->running) {
            try {
                $this->processDeliveryPayments();
                $this->updateOrderStatuses();
                $this->cleanupExpiredOrders();
                
                sleep(30);
                pcntl_signal_dispatch();
                
            } catch (Exception $e) {
                error_log("Erreur dans delivery_processor: " . $e->getMessage());
                sleep(60);
            }
        }
        
        echo "[" . date('Y-m-d H:i:s') . "] Arrêt du processeur de livraison\n";
    }
    
    private function processDeliveryPayments() {
        $commandes = db()->fetchAll("
            SELECT c.*, l.stripe_account_id, l.prenom, l.nom 
            FROM vr_commandes c 
            JOIN vr_livreurs l ON c.livreur_id = l.id 
            WHERE c.statut = 'livree' 
            AND c.stripe_transfer_id IS NULL 
            AND c.date_livraison <= DATE_SUB(NOW(), INTERVAL 2 MINUTE)
            AND l.stripe_account_id IS NOT NULL
        ");
        
        foreach ($commandes as $commande) {
            echo "[" . date('Y-m-d H:i:s') . "] Traitement paiement livreur pour commande #{$commande['id']}\n";
            
            $stripeHandler = new StripePaymentHandler();
            $result = $stripeHandler->transferToDriver(
                $commande['frais_livraison'],
                $commande['stripe_account_id'],
                'commande_' . $commande['id']
            );
            
            if ($result['success']) {
                db()->update('vr_commandes', 
                    ['stripe_transfer_id' => $result['transfer_id']], 
                    'id = ?', 
                    [$commande['id']]
                );
                
                $this->sendPaymentNotification($commande);
                echo "[" . date('Y-m-d H:i:s') . "] Paiement {$commande['frais_livraison']}€ envoyé à {$commande['prenom']} {$commande['nom']}\n";
                
            } else {
                error_log("Échec paiement livreur commande #{$commande['id']}: " . $result['error']);
            }
        }
    }
    
    private function updateOrderStatuses() {
        $commandesExpired = db()->fetchAll("
            SELECT c.*, r.first_name, r.last_name 
            FROM vr_commandes c 
            JOIN residents r ON c.resident_id = r.id 
            WHERE c.statut = 'en_attente' 
            AND c.date_commande <= DATE_SUB(NOW(), INTERVAL 30 MINUTE)
        ");
        
        foreach ($commandesExpired as $commande) {
            db()->update('vr_commandes', 
                ['statut' => 'annulee'], 
                'id = ?', 
                [$commande['id']]
            );
            
            if ($commande['stripe_payment_intent_id']) {
                $stripeHandler = new StripePaymentHandler();
                $stripeHandler->refundPayment($commande['stripe_payment_intent_id'], null, 'expired');
            }
            
            echo "[" . date('Y-m-d H:i:s') . "] Commande #{$commande['id']} expirée et annulée\n";
        }
        
        $commandesDelivering = db()->fetchAll("
            SELECT * FROM vr_commandes 
            WHERE statut IN ('en_route_client') 
            AND date_confirmation <= DATE_SUB(NOW(), INTERVAL 2 HOUR)
        ");
        
        foreach ($commandesDelivering as $commande) {
            $this->sendDelayNotification($commande);
        }
    }
    
    private function cleanupExpiredOrders() {
        db()->query("
            DELETE FROM vr_commandes 
            WHERE statut = 'annulee' 
            AND date_commande <= DATE_SUB(NOW(), INTERVAL 7 DAY)
        ");
        
        db()->query("
            DELETE FROM vr_messages 
            WHERE commande_id NOT IN (SELECT id FROM vr_commandes)
        ");
        
        db()->query("
            UPDATE vr_livreurs l 
            SET nb_commandes = (
                SELECT COUNT(*) FROM vr_commandes c 
                WHERE c.livreur_id = l.id AND c.statut = 'livree'
            )
        ");
    }
    
    private function sendPaymentNotification($commande) {
        $oneSignalAppId = db()->getConfig('onesignal_app_id');
        $oneSignalApiKey = db()->getConfig('onesignal_api_key');
        
        if ($oneSignalAppId && $oneSignalApiKey) {
            $notificationData = [
                'app_id' => $oneSignalAppId,
                'filters' => [
                    ['field' => 'tag', 'key' => 'livreur_id', 'relation' => '=', 'value' => $commande['livreur_id']]
                ],
                'headings' => ['fr' => 'Paiement reçu ! 💰'],
                'contents' => ['fr' => "Vous avez reçu {$commande['frais_livraison']}€ pour la commande #{$commande['id']}"],
                'data' => [
                    'type' => 'payment_received',
                    'amount' => $commande['frais_livraison']
                ]
            ];
            
            $this->sendNotification($notificationData);
        }
    }
    
    private function sendDelayNotification($commande) {
        $oneSignalAppId = db()->getConfig('onesignal_app_id');
        $oneSignalApiKey = db()->getConfig('onesignal_api_key');
        
        if ($oneSignalAppId && $oneSignalApiKey) {
            $notificationData = [
                'app_id' => $oneSignalAppId,
                'filters' => [
                    ['field' => 'tag', 'key' => 'client_id', 'relation' => '=', 'value' => $commande['resident_id']]
                ],
                'headings' => ['fr' => 'Livraison en retard'],
                'contents' => ['fr' => "Votre commande #{$commande['id']} prend plus de temps que prévu. Nous nous excusons pour le retard."],
                'data' => [
                    'type' => 'delivery_delay',
                    'commande_id' => $commande['id']
                ]
            ];
            
            $this->sendNotification($notificationData);
        }
    }
    
    private function sendNotification($data) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, 'https://onesignal.com/api/v1/notifications');
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Basic ' . db()->getConfig('onesignal_api_key')
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode !== 200) {
            error_log("Erreur envoi notification OneSignal: $httpCode - $response");
        }
    }
    
    public function signalHandler($signal) {
        echo "[" . date('Y-m-d H:i:s') . "] Signal reçu: $signal\n";
        $this->running = false;
    }
    
    public function shutdown() {
        echo "[" . date('Y-m-d H:i:s') . "] Arrêt propre du processeur\n";
    }
}

if (php_sapi_name() === 'cli') {
    $processor = new DeliveryProcessor();
    $processor->run();
}
?>