<?php
ini_set('session.gc_maxlifetime', 31536000);
ini_set('session.cookie_lifetime', 31536000);
session_set_cookie_params([
    'lifetime' => 31536000,
    'path' => '/',
    'domain' => $_SERVER['HTTP_HOST'],
    'secure' => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
    'httponly' => true,
    'samesite' => 'Lax'
]);
session_start();
include 'db.php';
include 'header-mobile.php';

// GESTION DE L'USAGE DE LA CARTE
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['track_card_usage'])) {
    ob_clean();
    header('Content-Type: application/json');
    
    if (!isset($_SESSION['resident_id'])) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Non authentifié']);
        exit;
    }
    
    $shop_id = intval($_POST['shop_id']);
    $resident_id = $_SESSION['resident_id'];
    
    try {
        $stmt = $conn->prepare("INSERT INTO card_usage (resident_id, shop_id, used_at) VALUES (?, ?, NOW())");
        $stmt->bind_param("ii", $resident_id, $shop_id);
        
        if (!$stmt->execute()) {
            throw new Exception('Erreur insertion card_usage');
        }
        
        $update_stmt = $conn->prepare("UPDATE residents SET points = points + 5 WHERE id = ?");
        $update_stmt->bind_param("i", $resident_id);
        
        if (!$update_stmt->execute()) {
            throw new Exception('Erreur mise à jour points');
        }
        
        $points_stmt = $conn->prepare("SELECT points FROM residents WHERE id = ?");
        $points_stmt->bind_param("i", $resident_id);
        $points_stmt->execute();
        $result = $points_stmt->get_result();
        $points = $result->fetch_assoc()['points'];
        
        $tiers = [
            ['name' => 'Cool', 'min' => 0, 'max' => 100],
            ['name' => 'Great', 'min' => 100, 'max' => 500, 'reward' => '10€ offert'],
            ['name' => 'Crown', 'min' => 500, 'max' => 1000, 'reward' => 'Panier gourmand 40€'],
            ['name' => 'Gold', 'min' => 1000, 'max' => 5000],
            ['name' => 'Hunter', 'min' => 5000, 'max' => PHP_INT_MAX]
        ];
        
        foreach ($tiers as $tier) {
            if ($points >= $tier['min'] && $points < $tier['max']) {
                $check = $conn->prepare("SELECT id FROM rewards WHERE resident_id = ? AND tier = ? AND status = 'pending'");
                $check->bind_param("is", $resident_id, $tier['name']);
                $check->execute();
                
                if ($check->get_result()->num_rows === 0 && isset($tier['reward'])) {
                    $reward_desc = $tier['reward'];
                    $insert = $conn->prepare("INSERT INTO rewards (resident_id, tier, points_required, reward_description) VALUES (?, ?, ?, ?)");
                    $insert->bind_param("isis", $resident_id, $tier['name'], $tier['min'], $reward_desc);
                    $insert->execute();
                }
                break;
            }
        }
        
        echo json_encode(['success' => true, 'points' => $points]);
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    
    exit;
}


$is_guest = false;
$resident = null;

if (!isset($_SESSION['resident_id']) && !isset($_SESSION['guest_mode'])) {
  header('Location: login.php');
  exit;
}

if (isset($_SESSION['guest_mode'])) {
  $is_guest = true;
} else {
  $resident_id = $_SESSION['resident_id'];
  $stmt = $conn->prepare("SELECT * FROM residents WHERE id = ?");
  $stmt->bind_param("i", $resident_id);
  $stmt->execute();
  $resident = $stmt->get_result()->fetch_assoc();
}

$is_member = !$is_guest;
$is_premium = false;

if (!$is_guest) {
  $is_premium = ($resident['member_'] == 1);
}

$offers_query = "SELECT * FROM resident_offers WHERE is_active = 1 AND valid_until >= NOW() ORDER BY created_at DESC";
$offers = mysqli_query($conn, $offers_query);

$success_message = '';
$error_message = '';

if (!$is_guest && isset($_POST['action'])) {
  if ($_POST['action'] == 'update_profile') {
      $first_name = trim($_POST['first_name']);
      $last_name = trim($_POST['last_name']);
      $email = trim($_POST['email']);
      $address = trim($_POST['address']);
      
      if (empty($first_name) || empty($last_name) || empty($email)) {
          $error_message = 'Tous les champs obligatoires doivent être remplis.';
      } else {
          $update_stmt = $conn->prepare("UPDATE residents SET first_name = ?, last_name = ?, email = ?, address = ? WHERE id = ?");
          $update_stmt->bind_param("ssssi", $first_name, $last_name, $email, $address, $resident_id);
          
          if ($update_stmt->execute()) {
              $success_message = 'Profil mis à jour avec succès !';
              $stmt = $conn->prepare("SELECT * FROM residents WHERE id = ?");
              $stmt->bind_param("i", $resident_id);
              $stmt->execute();
              $resident = $stmt->get_result()->fetch_assoc();
          } else {
              $error_message = 'Erreur lors de la mise à jour du profil.';
          }
      }
  }

  if ($_POST['action'] == 'change_password') {
      $current_password = $_POST['current_password'];
      $new_password = $_POST['new_password'];
      $confirm_password = $_POST['confirm_password'];
      
      if (password_verify($current_password, $resident['password'])) {
          if ($new_password === $confirm_password && strlen($new_password) >= 6) {
              $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
              $pwd_stmt = $conn->prepare("UPDATE residents SET password = ? WHERE id = ?");
              $pwd_stmt->bind_param("si", $hashed_password, $resident_id);
              
              if ($pwd_stmt->execute()) {
                  $success_message = 'Mot de passe modifié avec succès !';
              } else {
                  $error_message = 'Erreur lors de la modification du mot de passe.';
              }
          } else {
              $error_message = 'Le nouveau mot de passe doit contenir au moins 6 caractères et les mots de passe doivent correspondre.';
          }
      } else {
          $error_message = 'Mot de passe actuel incorrect.';
      }
  }
}

if (isset($_GET['action']) && $_GET['action'] == 'logout') {
  header('Location: logout.php');
  exit;
}

if (!$is_guest) {
   $member_since = new DateTime($resident['created_at']);
   $now = new DateTime();
   $months_diff = $now->diff($member_since)->m + ($now->diff($member_since)->y * 12);
   
   $badge = '<div class="wallet-badge wallet-badge-green">MEMBRE</div>';
   
   $address = htmlspecialchars($resident['address']);
   if (strlen($address) > 25) {
       $address = substr($address, 0, 22) . '...';
   }
}

$shops_query = "SELECT id, title, address_shop, latitude, longitude FROM shop WHERE active = 1";
$shops_result = mysqli_query($conn, $shops_query);
$shops = [];
while ($shop = mysqli_fetch_assoc($shops_result)) {
    $shops[] = $shop;
}
$shops_json = json_encode($shops);

$partners = [
    ['name' => 'Boulangerie Le Fournil', 'logo' => 'partner-boulangerie.png'],
    ['name' => 'Café des Amis', 'logo' => 'partner-cafe.png'],
    ['name' => 'Pharmacie Verte', 'logo' => 'partner-pharmacie.png'],
    ['name' => 'Restaurant La Table', 'logo' => 'partner-restaurant.png'],
    ['name' => 'Épicerie Bio Nature', 'logo' => 'partner-epicerie.png'],
    ['name' => 'Librairie du Quartier', 'logo' => 'partner-librairie.png']
];

$tiers = [
    ['name' => 'Cool', 'min' => 0, 'max' => 100, 'reward' => null],
    ['name' => 'Great', 'min' => 100, 'max' => 500, 'reward' => '10€ offert'],
    ['name' => 'Crown', 'min' => 500, 'max' => 1000, 'reward' => 'Panier gourmand 40€'],
    ['name' => 'Gold', 'min' => 1000, 'max' => 5000, 'reward' => null],
    ['name' => 'Hunter', 'min' => 5000, 'max' => PHP_INT_MAX, 'reward' => null]
];

function getCurrentTier($points, $tiers) {
    foreach ($tiers as $tier) {
        if ($points >= $tier['min'] && $points < $tier['max']) {
            return $tier;
        }
    }
    return end($tiers);
}

function getNextTier($points, $tiers) {
    foreach ($tiers as $tier) {
        if ($points < $tier['max']) {
            return $tier;
        }
    }
    return null;
}

$current_tier = getCurrentTier($resident['points'], $tiers);
$next_tier = getNextTier($resident['points'], $tiers);

if ($next_tier && $next_tier['min'] > $current_tier['min']) {
    $progress = (($resident['points'] - $current_tier['min']) / ($next_tier['min'] - $current_tier['min'])) * 100;
    $progress = min(100, max(0, $progress));
} else {
    $progress = 100;
}

$rewards_stmt = $conn->prepare("SELECT * FROM rewards WHERE resident_id = ? ORDER BY points_required DESC");
$rewards_stmt->bind_param("i", $resident_id);
$rewards_stmt->execute();
$rewards_result = $rewards_stmt->get_result();
?>

<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title><?= $is_guest ? 'Vert Chasseur' : 'Mon Compte - Vert Chasseur' ?></title>
  <link rel="icon" type="image/png" href="logo_new.png">
  <script src="https://cdn.tailwindcss.com"></script>
  <script>
      tailwind.config = {
          darkMode: 'class',
          theme: {
              extend: {}
          }
      }
  </script>
  <script>
  
  if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
    document.documentElement.classList.add('dark');
} else {
    document.documentElement.classList.remove('dark');
}

window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
    if (!localStorage.theme) {
        if (e.matches) {
            document.documentElement.classList.add('dark');
        } else {
            document.documentElement.classList.remove('dark');
        }
    }
});
  
  </script>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
      <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600&display=swap" rel="stylesheet">
  <link href="https://fonts.googleapis.com/css2?family=Princess+Sofia&display=swap" rel="stylesheet">
<style>
@media (min-width: 768px) {
    body {
        background-image: url('font-homepage.png');
        background-size: cover;
        background-position: center;
        background-attachment: fixed;
        background-repeat: no-repeat;
    }
}

@media (max-width: 767px) {
    body {
        background-color: #ffffff !important;
        background-image: none !important;
    }
    .dark body {
        background-color: #000000 !important;
        background-image: none !important;
    }
}

body {
    font-family: 'Poppins', sans-serif;
    letter-spacing: -0.01em;
}

.txt-chasseur {
    font-family: "Princess Sofia", serif;
    font-weight: 400;
}

.weather-banner {
    background-image: linear-gradient(
        45deg,
        hsl(355deg 70% 45%) 0%,
        hsl(0deg 68% 48%) 11%,
        hsl(5deg 66% 52%) 22%,
        hsl(10deg 64% 55%) 33%,
        hsl(15deg 62% 58%) 44%,
        hsl(20deg 60% 62%) 56%,
        hsl(15deg 62% 58%) 67%,
        hsl(10deg 64% 55%) 78%,
        hsl(5deg 66% 52%) 89%,
        hsl(0deg 68% 48%) 100%
    );
    border-radius: 16px;
    padding: 20px;
    color: white;
    cursor: pointer;
    transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
    box-shadow: 0 8px 32px rgba(70, 130, 180, 0.2);
}

.weather-banner:hover {
    transform: translateY(-4px);
    box-shadow: 0 12px 40px rgba(102, 126, 234, 0.3);
}

.weather-banner::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-image: 
        radial-gradient(circle at 20% 30%, rgba(255, 140, 0, 0.3) 1px, transparent 1px),
        radial-gradient(circle at 60% 70%, rgba(220, 20, 60, 0.3) 1px, transparent 1px),
        radial-gradient(circle at 80% 20%, rgba(255, 165, 0, 0.3) 1px, transparent 1px),
        radial-gradient(circle at 40% 80%, rgba(184, 92, 23, 0.3) 1px, transparent 1px),
        radial-gradient(circle at 90% 50%, rgba(210, 105, 30, 0.3) 1px, transparent 1px);
    background-size: 150px 150px, 200px 200px, 180px 180px, 220px 220px, 190px 190px;
    background-position: 0 0, 50px 50px, 25px 75px, 75px 25px, 100px 0;
    animation: fall 8s linear infinite;
    pointer-events: none;
}

@keyframes fall {
    0% {
        transform: translateY(-100%) rotate(0deg);
        opacity: 1;
    }
    100% {
        transform: translateY(100%) rotate(360deg);
        opacity: 0;
    }
}

.weather-banner:hover::before {
    top: -30%;
    right: -30%;
}

.weather-banner-content {
    position: relative;
    z-index: 2;
}

.weather-expanded {
    max-height: 0;
    overflow: hidden;
    transition: max-height 0.6s cubic-bezier(0.4, 0, 0.2, 1), opacity 0.4s ease 0.1s;
    opacity: 0;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 12px;
    margin-top: 0;
    backdrop-filter: blur(10px);
}

.weather-expanded.show {
    max-height: 800px;
    opacity: 1;
    margin-top: 24px;
}

.weather-expanded-inner {
    padding: 24px;
}

.hourly-forecast {
    display: flex;
    gap: 16px;
    overflow-x: auto;
    padding: 16px 0;
    scrollbar-width: thin;
    scrollbar-color: rgba(255, 255, 255, 0.3) transparent;
}

.hourly-forecast::-webkit-scrollbar {
    height: 6px;
}

.hourly-forecast::-webkit-scrollbar-track {
    background: rgba(255, 255, 255, 0.1);
    border-radius: 3px;
}

.hourly-forecast::-webkit-scrollbar-thumb {
    background: rgba(255, 255, 255, 0.3);
    border-radius: 3px;
}

.hour-item {
    flex-shrink: 0;
    text-align: center;
    min-width: 80px;
    padding: 16px 12px;
    background: rgba(255, 255, 255, 0.15);
    border-radius: 16px;
    backdrop-filter: blur(5px);
    transition: all 0.3s ease;
}

.hour-item:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-2px);
}

.forecast-day {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 16px;
    margin-bottom: 12px;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 16px;
    backdrop-filter: blur(5px);
    transition: all 0.3s ease;
}

.forecast-day:hover {
    background: rgba(255, 255, 255, 0.15);
    transform: translateX(4px);
}

.weather-source {
    font-size: 12px;
    opacity: 0.8;
    text-align: center;
    margin-top: 16px;
    font-style: italic;
}

.overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(255, 255, 255, 0.95);
    z-index: 10000;
    opacity: 0;
    visibility: hidden;
    transition: opacity 0.4s ease, visibility 0.4s ease;
    backdrop-filter: blur(10px);
}

.dark .overlay {
    background: rgba(0, 0, 0, 0.95);
}

.overlay.show {
    opacity: 1;
    visibility: visible;
}

.overlay-content {
    position: fixed;
    top: 0;
    right: -100%;
    width: 90%;
    max-width: 500px;
    height: 100%;
    background: white;
    box-shadow: -4px 0 24px rgba(0, 0, 0, 0.15);
    overflow-y: auto;
    padding: 20px;
    transition: right 0.4s cubic-bezier(0.4, 0, 0.2, 1);
}

.dark .overlay-content {
    background: black;
    box-shadow: -4px 0 24px rgba(255, 255, 255, 0.1);
}

.overlay.show .overlay-content {
    right: 0;
}

.overlay-content-left {
    position: fixed;
    top: 0;
    left: -100%;
    width: 90%;
    max-width: 500px;
    height: 100%;
    background: white;
    box-shadow: 4px 0 24px rgba(0, 0, 0, 0.15);
    overflow-y: auto;
    padding: 20px;
    transition: left 0.4s cubic-bezier(0.4, 0, 0.2, 1);
}

.dark .overlay-content-left {
    background: black;
    box-shadow: 4px 0 24px rgba(255, 255, 255, 0.1);
}

.overlay.show .overlay-content-left {
    left: 0;
}

.overlay-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 32px;
    padding-top: 20px;
}

.overlay-title {
    font-size: 32px;
    font-weight: 300;
    color: #000000;
    letter-spacing: -0.02em;
}

.dark .overlay-title {
    color: #ffffff;
}

.overlay-close {
    width: 48px;
    height: 48px;
    background: rgba(0, 0, 0, 0.1);
    border: none;
    border-radius: 50%;
    color: #000000;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: background 0.2s ease;
}

.dark .overlay-close {
    background: rgba(255, 255, 255, 0.1);
    color: #ffffff;
}

.overlay-close:hover {
    background: rgba(0, 0, 0, 0.2);
}

.dark .overlay-close:hover {
    background: rgba(255, 255, 255, 0.2);
}

.tabs {
    display: flex;
    gap: 4px;
    margin-bottom: 32px;
    background: rgba(0, 0, 0, 0.05);
    padding: 4px;
    border-radius: 12px;
    flex-wrap: wrap;
}

.dark .tabs {
    background: rgba(255, 255, 255, 0.05);
}

.tab-btn {
    flex: 1;
    min-width: 80px;
    padding: 12px 16px;
    background: transparent;
    border: none;
    border-radius: 8px;
    color: rgba(0, 0, 0, 0.7);
    font-size: 16px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.dark .tab-btn {
    color: rgba(255, 255, 255, 0.7);
}

.tab-btn.active {
    background: rgba(0, 0, 0, 0.15);
    color: #000000;
}

.dark .tab-btn.active {
    background: rgba(255, 255, 255, 0.15);
    color: #ffffff;
}

.tab-btn:hover:not(.active) {
    background: rgba(0, 0, 0, 0.08);
    color: rgba(0, 0, 0, 0.9);
}

.dark .tab-btn:hover:not(.active) {
    background: rgba(255, 255, 255, 0.08);
    color: rgba(255, 255, 255, 0.9);
}

.tab-content {
    display: none;
    opacity: 0;
    transform: translateY(10px);
    transition: opacity 0.3s ease, transform 0.3s ease;
}

.tab-content.active {
    display: block;
    animation: tabFadeIn 0.3s ease forwards;
}

@keyframes tabFadeIn {
    from {
        opacity: 0;
        transform: translateY(10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.form-group {
    margin-bottom: 24px;
}

.form-label {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: rgba(0, 0, 0, 0.9);
    margin-bottom: 8px;
}

.dark .form-label {
    color: rgba(255, 255, 255, 0.9);
}

.form-input {
    width: 100%;
    padding: 16px;
    background: rgba(0, 0, 0, 0.05);
    border: 1px solid rgba(0, 0, 0, 0.1);
    border-radius: 12px;
    color: #000000;
    font-size: 16px;
    transition: all 0.2s ease;
}

.dark .form-input {
    background: rgba(255, 255, 255, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.2);
    color: #ffffff;
}

.form-input::placeholder {
    color: rgba(0, 0, 0, 0.4);
}

.dark .form-input::placeholder {
    color: rgba(255, 255, 255, 0.5);
}

.form-input:focus {
    outline: none;
    border-color: #16a34a;
    background: rgba(0, 0, 0, 0.08);
}

.dark .form-input:focus {
    background: rgba(255, 255, 255, 0.15);
}

.form-textarea {
    resize: vertical;
    min-height: 100px;
}

.btn-primary {
    width: 100%;
    padding: 16px;
    background: #16a34a;
    border: none;
    border-radius: 12px;
    color: #ffffff;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: background 0.2s ease;
}

.btn-primary:hover {
    background: #15803d;
}

.btn-secondary {
    width: 100%;
    padding: 16px;
    background: #6b7280;
    border: none;
    border-radius: 12px;
    color: #ffffff;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: background 0.2s ease;
    margin-bottom: 12px;
}

.btn-secondary:hover {
    background: #4b5563;
}

.btn-danger {
    background: #dc2626;
    padding: 12px 20px;
    border: none;
    border-radius: 8px;
    color: #ffffff;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: background 0.2s ease;
    margin-right: 12px;
    margin-bottom: 8px;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.btn-danger:hover {
    background: #b91c1c;
}

.alert {
    padding: 16px;
    border-radius: 12px;
    margin-bottom: 24px;
    font-size: 14px;
    font-weight: 500;
}

.alert-success {
    background: rgba(34, 197, 94, 0.1);
    border: 1px solid rgba(34, 197, 94, 0.3);
    color: #15803d;
}

.dark .alert-success {
    color: #4ade80;
}

.alert-error {
    background: rgba(239, 68, 68, 0.1);
    border: 1px solid rgba(239, 68, 68, 0.3);
    color: #dc2626;
}

.dark .alert-error {
    color: #f87171;
}

.info-card {
    background: rgba(0, 0, 0, 0.03);
    border: 1px solid rgba(0, 0, 0, 0.1);
    border-radius: 16px;
    padding: 24px;
    margin-bottom: 24px;
}

.dark .info-card {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.info-card h3 {
    font-size: 20px;
    font-weight: 600;
    color: #000000;
    margin-bottom: 16px;
}

.dark .info-card h3 {
    color: #ffffff;
}

.info-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 0;
    border-bottom: 1px solid rgba(0, 0, 0, 0.1);
}

.dark .info-item {
    border-bottom: 1px solid rgba(255, 255, 255, 0.1);
}

.info-item:last-child {
    border-bottom: none;
}

.info-label {
    font-size: 14px;
    font-weight: 500;
    color: rgba(0, 0, 0, 0.7);
}

.dark .info-label {
    color: rgba(255, 255, 255, 0.7);
}

.info-value {
    font-size: 14px;
    color: #000000;
    font-weight: 500;
}

.dark .info-value {
    color: #ffffff;
}

.danger-zone {
    background: rgba(239, 68, 68, 0.1);
    border: 1px solid rgba(239, 68, 68, 0.3);
    border-radius: 16px;
    padding: 24px;
}

.danger-zone h3 {
    font-size: 20px;
    font-weight: 600;
    color: #dc2626;
    margin-bottom: 8px;
}

.dark .danger-zone h3 {
    color: #f87171;
}

.danger-zone p {
    color: rgba(220, 38, 38, 0.8);
    font-size: 14px;
    margin-bottom: 20px;
}

.dark .danger-zone p {
    color: rgba(248, 113, 113, 0.8);
}

.wallet-card {
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
    border-radius: 24px;
    padding: 32px;
    position: relative;
    overflow: hidden;
    margin-bottom: 32px;
    max-width: 400px;
    margin-left: auto;
    margin-right: auto;
}

.wallet-card::before {
    content: '';
    position: absolute;
    top: -50px;
    right: -50px;
    width: 120px;
    height: 120px;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 50%;
}

.wallet-badge {
    position: absolute;
    top: 16px;
    right: 16px;
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 11px;
    font-weight: 700;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    z-index: 10;
}

.wallet-badge-green {
    background: #22c55e;
    color: #ffffff;
}

.wallet-content {
    position: relative;
    z-index: 2;
    display: flex;
    flex-direction: column;
    color: #ffffff;
}

.wallet-header {
    text-align: center;
    margin-bottom: 32px;
}

.wallet-name {
    font-size: 24px;
    font-weight: 700;
    line-height: 1.2;
    margin-bottom: 8px;
}

.wallet-title {
    font-size: 16px;
    opacity: 0.9;
    font-weight: 500;
}

.wallet-qr-center {
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 32px 0;
}

.qr-container-center {
    background: white;
    padding: 20px;
    border-radius: 20px;
    box-shadow: 0 8px 24px rgba(0, 0, 0, 0.2);
}

.qr-img-center {
    display: block;
    width: 180px;
    height: 180px;
}

.wallet-footer {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: 32px;
    padding-top: 24px;
    border-top: 1px solid rgba(255, 255, 255, 0.2);
}

.wallet-id {
    font-size: 13px;
    opacity: 0.9;
    font-weight: 600;
}

.wallet-address {
    font-size: 12px;
    opacity: 0.85;
    font-weight: 500;
    text-align: right;
    max-width: 150px;
    line-height: 1.4;
}

.wallet-card-disabled {
    background: linear-gradient(135deg, #9ca3af 0%, #6b7280 100%);
    opacity: 0.9;
}

.wallet-card-disabled .wallet-content {
    min-height: 280px;
}

.subscription-prompt {
    text-align: center;
    padding: 32px;
}

.subscription-prompt h3 {
    font-size: 20px;
    font-weight: 600;
    margin-bottom: 12px;
}

.subscription-prompt p {
    font-size: 14px;
    opacity: 0.95;
    margin-bottom: 24px;
    line-height: 1.5;
}

.subscription-price {
    font-size: 32px;
    font-weight: 700;
    margin-bottom: 8px;
}

.subscription-period {
    font-size: 14px;
    opacity: 0.8;
    margin-bottom: 24px;
}

.btn-subscribe {
    width: 100%;
    padding: 16px;
    background: white;
    color: #374151;
    border: none;
    border-radius: 12px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    margin-bottom: 12px;
}

.btn-subscribe:hover {
    background: #f3f4f6;
    transform: translateY(-2px);
}

.btn-refresh {
    width: 100%;
    padding: 12px;
    background: rgba(255, 255, 255, 0.2);
    color: white;
    border: 1px solid rgba(255, 255, 255, 0.3);
    border-radius: 12px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
}

.btn-refresh:hover {
    background: rgba(255, 255, 255, 0.3);
}

.filter-button {
    padding: 8px 16px;
    border-radius: 20px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    border: 1px solid #e5e7eb;
    background: transparent;
    color: #6b7280;
}

.dark .filter-button {
    border-color: #374151;
    color: #9ca3af;
}

.filter-button.active {
    background: #ffffff;
    color: #1f2937;
    border-color: #d1d5db;
}

.dark .filter-button.active {
    background: #000000;
    color: #f9fafb;
    border-color: #374151;
}

.filter-button:hover:not(.active) {
    background: #f9fafb;
    border-color: #d1d5db;
}

.dark .filter-button:hover:not(.active) {
    background: #111827;
    border-color: #374151;
}

.partners-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
    gap: 16px;
    margin-top: 16px;
}

.partner-item {
    display: flex;
    flex-direction: column;
    align-items: center;
    padding: 16px;
    background: rgba(0, 0, 0, 0.03);
    border: 1px solid rgba(0, 0, 0, 0.1);
    border-radius: 12px;
    transition: all 0.2s ease;
}

.dark .partner-item {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.partner-item:hover {
    background: rgba(0, 0, 0, 0.05);
    transform: translateY(-2px);
}

.dark .partner-item:hover {
    background: rgba(255, 255, 255, 0.08);
}

.partner-logo {
    width: 40px;
    height: 40px;
    background: #16a34a;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 8px;
    font-weight: 700;
    color: white;
    font-size: 12px;
}

.partner-name {
    font-size: 12px;
    font-weight: 500;
    text-align: center;
    color: #000000;
    line-height: 1.3;
}

.dark .partner-name {
    color: #ffffff;
}

.overlay-description {
    font-size: 0.875rem;
    opacity: 0.7;
    margin-top: 0.25rem;
    font-weight: 300;
    color: #000;
}

.dark .overlay-description {
    color: #fff;
}

.help-option {
    background: rgba(0, 0, 0, 0.03);
    border: 1px solid rgba(0, 0, 0, 0.1);
    border-radius: 16px;
    padding: 20px;
    margin-bottom: 16px;
    transition: all 0.2s ease;
    cursor: pointer;
}

.dark .help-option {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.help-option:hover {
    background: rgba(0, 0, 0, 0.05);
    transform: translateX(4px);
}

.dark .help-option:hover {
    background: rgba(255, 255, 255, 0.08);
}

.help-icon {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 12px;
}

.help-icon.green {
    background: rgba(22, 163, 74, 0.1);
    color: #16a34a;
}

.help-icon.blue {
    background: rgba(59, 130, 246, 0.1);
    color: #3b82f6;
}

.help-icon.purple {
    background: rgba(168, 85, 247, 0.1);
    color: #a855f7;
}

.card-overlay-blur {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(8px);
    z-index: 15000;
    opacity: 0;
    visibility: hidden;
    transition: opacity 0.3s ease, visibility 0.3s ease;
}

.card-overlay-blur.show {
    opacity: 1;
    visibility: visible;
}

.card-confirmation-dialog {
    position: fixed;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%) scale(0.9);
    background: white;
    border-radius: 20px;
    padding: 32px;
    max-width: 400px;
    width: 90%;
    z-index: 15001;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s ease;
}

.dark .card-confirmation-dialog {
    background: #000000;
}

.card-overlay-blur.show ~ .card-confirmation-dialog {
    opacity: 1;
    visibility: visible;
    transform: translate(-50%, -50%) scale(1);
}

.card-dialog-title {
    font-size: 20px;
    font-weight: 600;
    margin-bottom: 12px;
    color: #000000;
}

.dark .card-dialog-title {
    color: #ffffff;
}

.card-dialog-text {
    font-size: 16px;
    color: #666666;
    margin-bottom: 24px;
    line-height: 1.5;
}

.dark .card-dialog-text {
    color: #999999;
}

.card-dialog-buttons {
    display: flex;
    gap: 12px;
}

.card-dialog-btn {
    flex: 1;
    padding: 12px 16px;
    border: none;
    border-radius: 12px;
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
}

.card-dialog-btn-yes {
    background: #16a34a;
    color: white;
}

.card-dialog-btn-yes:hover {
    background: #15803d;
}

.card-dialog-btn-no {
    background: rgba(0, 0, 0, 0.1);
    color: #000000;
}

.dark .card-dialog-btn-no {
    background: rgba(255, 255, 255, 0.1);
    color: #ffffff;
}

.card-dialog-btn-no:hover {
    background: rgba(0, 0, 0, 0.15);
}

.dark .card-dialog-btn-no:hover {
    background: rgba(255, 255, 255, 0.15);
}

.card-display-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.95);
    z-index: 16000;
    display: flex;
    align-items: center;
    justify-content: center;
    opacity: 0;
    visibility: hidden;
    transition: opacity 0.3s ease, visibility 0.3s ease;
}

.card-display-overlay.show {
    opacity: 1;
    visibility: visible;
}

.card-display-close {
    position: absolute;
    top: 20px;
    right: 20px;
    width: 48px;
    height: 48px;
    background: rgba(255, 255, 255, 0.2);
    border: none;
    border-radius: 50%;
    cursor: pointer;
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 16001;
    transition: background 0.2s ease;
}

.card-display-close:hover {
    background: rgba(255, 255, 255, 0.3);
}

.card-display-content {
    width: 100%;
    max-width: 500px;
    padding: 20px;
}

@media (max-width: 768px) {       
    .partners-grid {
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 12px;
    }

    .weather-banner {
        border-radius: 12px;
        padding: 16px;
    }

    .weather-expanded-inner {
        padding: 16px;
    }

    .hour-item {
        min-width: 70px;
        padding: 12px 8px;
    }

    .overlay-content, .overlay-content-left {
        width: 100%;
        max-width: 100%;
    }
}

@media (max-width: 640px) {
    .overlay-content, .overlay-content-left {
        padding: 16px;
    }
    
    .overlay-title {
        font-size: 24px;
    }
    
    .wallet-card {
        padding: 24px;
        margin-bottom: 24px;
    }
    
    .wallet-name {
        font-size: 20px;
    }
    
    .wallet-title {
        font-size: 14px;
    }
    
    .qr-container-center {
        padding: 16px;
    }
    
    .qr-img-center {
        width: 150px;
        height: 150px;
    }
    
    .wallet-footer {
        margin-top: 24px;
        padding-top: 20px;
    }
    
    .partners-grid {
        grid-template-columns: repeat(2, 1fr);
    }

    .card-confirmation-dialog {
        padding: 24px;
        max-width: 90vw;
    }
}
</style>
</head>

<body class="bg-white dark:bg-black transition-colors duration-300">
  <script>
      if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
          document.documentElement.classList.add('dark');
      } else {
          document.documentElement.classList.remove('dark');
      }
  </script>
  
 <style>
.welcome-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100vw;
    height: 100vh;
    background: white;
    z-index: 50000;
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s ease;
    overflow-y: auto;
    overflow-x: hidden;
    -webkit-overflow-scrolling: touch;
}

.dark .welcome-overlay {
    background: black;
}

.welcome-overlay.show {
    opacity: 1;
    visibility: visible;
}

.welcome-content {
    min-height: 100vh;
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    padding: 80px 20px 40px;
    position: relative;
}

.welcome-close {
    position: fixed;
    top: 20px;
    right: 20px;
    width: 44px;
    height: 44px;
    border: none;
    background: rgba(0, 0, 0, 0.08);
    border-radius: 50%;
    cursor: pointer;
    font-size: 24px;
    color: #000;
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 51000;
    transition: all 0.2s ease;
}

.dark .welcome-close {
    background: rgba(255, 255, 255, 0.08);
    color: #fff;
}

.welcome-close:hover {
    background: rgba(0, 0, 0, 0.12);
    transform: scale(1.05);
}

.dark .welcome-close:hover {
    background: rgba(255, 255, 255, 0.12);
}

.welcome-header {
    text-align: center;
    margin-bottom: 48px;
}

.welcome-title {
    font-size: 36px;
    font-weight: 300;
    color: #000;
    margin-bottom: 16px;
    line-height: 1.2;
}

.dark .welcome-title {
    color: #fff;
}

.welcome-subtitle {
    font-size: 18px;
    color: #16a34a;
    font-weight: 500;
}

.welcome-list {
    max-width: 640px;
    width: 100%;
}

.welcome-item {
    display: flex;
    align-items: flex-start;
    gap: 16px;
    padding: 24px 0;
    border-bottom: 1px solid rgba(0, 0, 0, 0.08);
}

.dark .welcome-item {
    border-bottom: 1px solid rgba(255, 255, 255, 0.08);
}

.welcome-item:last-child {
    border-bottom: none;
}

.welcome-icon {
    width: 24px;
    height: 24px;
    color: #16a34a;
    flex-shrink: 0;
    margin-top: 2px;
}

.welcome-item-content h3 {
    font-size: 18px;
    font-weight: 600;
    margin-bottom: 8px;
    color: #000;
}

.dark .welcome-item-content h3 {
    color: #fff;
}

.welcome-item-content p {
    font-size: 16px;
    color: #666;
    line-height: 1.6;
}

.dark .welcome-item-content p {
    color: #999;
}

.welcome-actions {
    margin-top: 48px;
    text-align: center;
    width: 100%;
    max-width: 640px;
}

.welcome-btn {
    padding: 16px 40px;
    background: #16a34a;
    color: white;
    border: none;
    border-radius: 12px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    width: 100%;
    max-width: 300px;
}

.welcome-btn:hover {
    background: #15803d;
    transform: translateY(-2px);
}

@media (max-width: 768px) {
    .welcome-content {
        padding: 60px 24px 40px;
        justify-content: flex-start;
    }
    
    .welcome-header {
        margin-bottom: 40px;
        margin-top: 20px;
    }
    
    .welcome-title {
        font-size: 28px;
    }
    
    .welcome-subtitle {
        font-size: 16px;
    }
    
    .welcome-item {
        padding: 20px 0;
    }
    
    .welcome-item-content h3 {
        font-size: 17px;
    }
    
    .welcome-item-content p {
        font-size: 15px;
    }
    
    .welcome-actions {
        margin-top: 40px;
    }
}

@media (max-width: 480px) {
    .welcome-content {
        padding: 60px 16px 32px;
    }
    
    .welcome-close {
        width: 40px;
        height: 40px;
        top: 16px;
        right: 16px;
    }
    
    .welcome-header {
        margin-bottom: 32px;
    }
    
    .welcome-title {
        font-size: 24px;
    }
    
    .welcome-subtitle {
        font-size: 15px;
    }
    
    .welcome-item {
        padding: 18px 0;
        gap: 12px;
    }
    
    .welcome-icon {
        width: 22px;
        height: 22px;
    }
    
    .welcome-item-content h3 {
        font-size: 16px;
    }
    
    .welcome-item-content p {
        font-size: 14px;
    }
    
    .welcome-actions {
        margin-top: 32px;
    }
    
    .welcome-btn {
        padding: 14px 32px;
        font-size: 15px;
    }
}

@media (min-width: 768px) {
    body {
        background-image: none;
        background-color: #87CEEB;
        position: relative;
        overflow-x: hidden;
    }
    
    .dark body {
        background-color: #0F1729;
    }
}


@keyframes fall {
    0% {
        transform: translateY(-10px) translateX(0) rotate(0deg);
        opacity: 0.3;
    }
    50% {
        opacity: 0.2;
    }
    100% {
        transform: translateY(100vh) translateX(20px) rotate(360deg);
        opacity: 0;
    }
}

main {
    position: relative;
    z-index: 2;
}
</style>
<!--- SEULEMENT POUR LES FETES DE FIN D'ANNEE ---->

<script>
function createSnowflakes() {
    const snowflakesContainer = document.createElement('div');
    snowflakesContainer.className = 'snowflakes';
    document.body.appendChild(snowflakesContainer);
    
    for (let i = 0; i < 80; i++) {
        const snowflake = document.createElement('div');
        snowflake.className = 'snowflake';
        snowflake.style.left = Math.random() * 100 + '%';
        snowflake.style.width = Math.random() * 4 + 2 + 'px';
        snowflake.style.height = snowflake.style.width;
        snowflake.style.animationDuration = Math.random() * 3 + 6 + 's';
        snowflake.style.animationDelay = Math.random() * 5 + 's';
        snowflakesContainer.appendChild(snowflake);
    }
}

document.addEventListener('DOMContentLoaded', createSnowflakes);
</script>
<!------->
<div id="welcomeOverlay" class="welcome-overlay">
    <button onclick="closeWelcome()" class="welcome-close">×</button>
    <div class="welcome-content">
        <div class="welcome-header">
            <h1 class="welcome-title">
                Bienvenue à <span class="txt-chasseur text-green-600">Vert Chasseur</span>
            </h1>
            <p class="welcome-subtitle">Avant toute chose, voici comment fonctionne l'application</p>
        </div>
        
        <div class="welcome-list">
            <div class="welcome-item">
                <svg class="welcome-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"/>
                </svg>
                <div class="welcome-item-content">
                    <h3>Page d'accueil</h3>
                    <p>Consulte la météo ultra-précise du quartier, lis le poème de la semaine et découvre les offres spéciales de nos partenaires locaux.</p>
                </div>
            </div>
            
            <div class="welcome-item">
                <svg class="welcome-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 20H5a2 2 0 01-2-2V6a2 2 0 012-2h10a2 2 0 012 2v1m2 13a2 2 0 01-2-2V7m2 13a2 2 0 002-2V9a2 2 0 00-2-2h-2m-4-3H9M7 16h6M7 8h6v4H7V8z"/>
                </svg>
                <div class="welcome-item-content">
                    <h3>Page des actualités</h3>
                    <p>Reste informé sur tout ce qui se passe dans le quartier. Événements, annonces importantes et nouveautés locales.</p>
                </div>
            </div>
            
            <div class="welcome-item">
                <svg class="welcome-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 11V7a4 4 0 00-8 0v4M5 9h14l1 12H4L5 9z"/>
                </svg>
                <div class="welcome-item-content">
                    <h3>Page des commerces</h3>
                    <p>Explore tous les commerces partenaires du quartier et profite de réductions exclusives avec ta carte membre.</p>
                </div>
            </div>
            
            <div class="welcome-item">
                <svg class="welcome-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/>
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                </svg>
                <div class="welcome-item-content">
                    <h3>Paramètres</h3>
                    <p>Accède à ta carte membre dans l'onglet Carte, modifie ton profil, change ton mot de passe et gère tes préférences.</p>
                </div>
            </div>
        </div>
        
        <div class="welcome-actions">
            <button onclick="closeWelcome()" class="welcome-btn">
                J'ai compris
            </button>
        </div>
    </div>
</div>

<script>
function showWelcome() {
    document.getElementById('welcomeOverlay').classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closeWelcome() {
    document.getElementById('welcomeOverlay').classList.remove('show');
    document.body.style.overflow = 'auto';
    localStorage.setItem('vert_chasseur_welcome', 'seen');
}

document.addEventListener('DOMContentLoaded', function() {
    if (!localStorage.getItem('vert_chasseur_welcome')) {
        setTimeout(showWelcome, 800);
    }
});
</script>

<script>
const maintenanceConfig = {
    enabled: true,
    logoLight: 'logo_new.png',
    logoDark: 'logo_dark.png',
    title: 'Évolutions en cours',
    description: 'Notre application traverse actuellement une phase d\'amélioration. Des perturbations passagères peuvent survenir durant votre navigation. La fonction de souscription à la carte membre demeure temporairement indisponible le temps de sa configuration.',
    buttonText: 'Accéder malgré tout'
};

function initMaintenance() {
    const overlay = document.getElementById('maintenanceOverlay');
    const bg = overlay.querySelector('div[style*="opacity"]');
    const logo = document.getElementById('maintenanceLogo');
    const title = document.getElementById('maintenanceTitle');
    const desc = document.getElementById('maintenanceDesc');
    const btn = document.getElementById('maintenanceBtn');
    
    const isDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
    
    if (isDark) {
        bg.style.backgroundColor = '#000';
        title.style.color = '#fff';
        desc.style.color = '#fff';
        btn.style.backgroundColor = '#fff';
        btn.style.color = '#000';
        logo.src = maintenanceConfig.logoDark;
    } else {
        bg.style.backgroundColor = '#fff';
        title.style.color = '#000';
        desc.style.color = '#000';
        btn.style.backgroundColor = '#000';
        btn.style.color = '#fff';
        logo.src = maintenanceConfig.logoLight;
    }
    
    title.textContent = maintenanceConfig.title;
    desc.textContent = maintenanceConfig.description;
    btn.textContent = maintenanceConfig.buttonText;
    
    btn.addEventListener('click', () => {
        overlay.style.display = 'none';
    });
    
    btn.addEventListener('mouseenter', () => {
        btn.style.transform = 'translateY(-2px)';
        btn.style.opacity = '0.9';
    });
    
    btn.addEventListener('mouseleave', () => {
        btn.style.transform = 'translateY(0)';
        btn.style.opacity = '1';
    });
    
    if (maintenanceConfig.enabled) {
        overlay.style.display = 'block';
    }
}

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initMaintenance);
} else {
    initMaintenance();
}
</script>

<div id="cardOverlayBlur" class="card-overlay-blur"></div>
<div id="cardConfirmationDialog" class="card-confirmation-dialog">
    <h2 class="card-dialog-title">Utiliser la carte ?</h2>
    <p class="card-dialog-text" id="cardDialogMessage"></p>
    <div class="card-dialog-buttons">
        <button class="card-dialog-btn card-dialog-btn-no" onclick="closeCardDialog()">Non</button>
        <button class="card-dialog-btn card-dialog-btn-yes" onclick="confirmCardUsage()">Oui</button>
    </div>
</div>

<div id="cardDisplayOverlay" class="card-display-overlay">
    <button class="card-display-close" onclick="closeCardDisplay()">×</button>
    <div class="card-display-content">
        <div class="wallet-card">
            <?php if (!$is_guest): ?><?= $badge ?><?php endif; ?>
            <div class="wallet-content">
                <div class="wallet-header">
                    <h3 class="wallet-name"><?php if (!$is_guest): ?><?= htmlspecialchars($resident['first_name'] . ' ' . $resident['last_name']) ?><?php endif; ?></h3>
                    <p class="wallet-title">Membre Vert Chasseur</p>
                </div>
                <div class="wallet-qr-center">
                    <div class="qr-container-center">
                        <img src="https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=<?php if (!$is_guest): ?><?= urlencode($resident['id']) ?><?php endif; ?>" alt="QR Code" class="qr-img-center">
                    </div>
                </div>
                <div class="wallet-footer">
                    <div class="wallet-id"><?php if (!$is_guest): ?>ID: <?= $resident['id'] ?><?php endif; ?></div>
                    <div class="wallet-address"><?php if (!$is_guest): ?><?= $address ?><?php endif; ?></div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php if (!$is_guest): ?>
<?php
$subscription_months = 0;

if ($is_premium) {
    $sub_stmt = $conn->prepare("SELECT created_at FROM subscriptions WHERE resident_id = ? AND status = 'active'");
    $sub_stmt->bind_param("i", $resident_id);
    $sub_stmt->execute();
    $sub_result = $sub_stmt->get_result();
    
    if ($sub_row = $sub_result->fetch_assoc()) {
        $sub_start = new DateTime($sub_row['created_at']);
        $now = new DateTime();
        $interval = $now->diff($sub_start);
        $subscription_months = ($interval->y * 12) + $interval->m;
    }
}
?>

<div id="helpOverlay" class="overlay">
   <div class="overlay-content-left">
       <div class="max-w-2xl mx-auto">
           <div class="overlay-header">
               <div>
                   <h1 class="overlay-title">Aide</h1>
                   <p class="overlay-description">Besoin d'assistance ?<br>Nous sommes là pour t'accompagner</p>
               </div>
               <button onclick="closeHelp()" class="overlay-close">
                   <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                       <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                   </svg>
               </button>
           </div>

           <div class="space-y-4">
            <!---   <a href="tel:+3225551234" class="help-option block">
                   <div class="help-icon green">
                       <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                           <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 5a2 2 0 012-2h3.28a1 1 0 01.948.684l1.498 4.493a1 1 0 01-.502 1.21l-2.257 1.13a11.042 11.042 0 005.516 5.516l1.13-2.257a1 1 0 011.21-.502l4.493 1.498a1 1 0 01.684.949V19a2 2 0 01-2 2h-1C9.716 21 3 14.284 3 6V5z"/>
                       </svg>
                   </div>
                   <h3 class="text-lg font-semibold text-stone-900 dark:text-white mb-2">Assistance Vert Chasseur</h3>
                   <p class="text-sm text-stone-600 dark:text-stone-400 mb-3">Pour toute question concernant l'application, ta carte membre ou les partenaires du quartier.</p>
                   <div class="text-green-600 dark:text-green-400 font-semibold">+32 2 555 12 34</div>
                   <p class="text-xs text-stone-500 dark:text-stone-500 mt-1">Lundi - Vendredi : 9h - 18h</p>
               </a> --->

               <a href="tel:112" class="help-option block">
                   <div class="help-icon blue">
                       <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                           <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                       </svg>
                   </div>
                   <h3 class="text-lg font-semibold text-stone-900 dark:text-white mb-2">Urgences locales</h3>
                   <p class="text-sm text-stone-600 dark:text-stone-400 mb-3">Un incident ? Contacte immédiatement les services d'urgence.</p>
                   <div class="text-blue-600 dark:text-blue-400 font-semibold">112</div>
                   <p class="text-xs text-stone-500 dark:text-stone-500 mt-1">Disponible 24h/24 - 7j/7</p>
               </a>

               <div onclick="openTicketSystem()" class="help-option">
                   <div class="help-icon purple">
                       <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                           <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                       </svg>
                   </div>
                   <h3 class="text-lg font-semibold text-stone-900 dark:text-white mb-2">Service de tickets</h3>
                   <p class="text-sm text-stone-600 dark:text-stone-400 mb-3">Pour les demandes professionnelles, suggestions d'amélioration ou partenariats.</p>
                   <div class="text-purple-600 dark:text-purple-400 font-semibold">Créer un ticket</div>
                   <p class="text-xs text-stone-500 dark:text-stone-500 mt-1">Réponse sous 48h ouvrées</p>
               </div>

               <div class="info-card">
                   <h3>Informations utiles</h3>
                   <div class="space-y-3 text-sm text-stone-700 dark:text-stone-300">
                       <div>
                           <strong>Email pour demande professionnelle :</strong><br>
                           msg@aktascorp.com
                       </div>
                   </div>
               </div>
           </div>
       </div>
   </div>
</div>


<script>
const cardColors = {
    green: 'linear-gradient(135deg, #059669 0%, #047857 100%)',
    forest: 'linear-gradient(135deg, #065f46 0%, #064e3b 100%)',
    sage: 'linear-gradient(135deg, #6b9080 0%, #4f6f5c 100%)',
    mint: 'linear-gradient(135deg, #a7c4bc 0%, #84a59d 100%)',
    olive: 'linear-gradient(135deg, #8a9a5b 0%, #6b7c3d 100%)',
    pine: 'linear-gradient(135deg, #2d5016 0%, #1e3a0f 100%)'
};

function changeCardColor(color) {
    document.querySelectorAll('.color-check').forEach(check => check.textContent = '');
    document.getElementById('check-' + color).textContent = '✓';
    
    const previewCard = document.getElementById('preview-card');
    if (previewCard) {
        previewCard.style.background = cardColors[color];
    }
    
    document.querySelectorAll('.wallet-card').forEach(card => {
        if (!card.classList.contains('wallet-card-disabled') && card.id !== 'preview-card') {
            card.style.background = cardColors[color];
        }
    });
    
    localStorage.setItem('vert_chasseur_card_color', color);
}

document.addEventListener('DOMContentLoaded', function() {
    const savedColor = localStorage.getItem('vert_chasseur_card_color') || 'green';
    changeCardColor(savedColor);
});
</script>
<style>

.card-color-btn {
    aspect-ratio: 1;
    border-radius: 12px;
    border: 3px solid transparent;
    cursor: pointer;
    transition: all 0.2s ease;
    position: relative;
    min-height: 60px;
}

.card-color-btn:hover {
    transform: scale(1.05);
    border-color: rgba(0, 0, 0, 0.2);
}

.dark .card-color-btn:hover {
    border-color: rgba(255, 255, 255, 0.2);
}

.color-check {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    color: white;
    font-size: 24px;
    font-weight: bold;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.5);
}
    
    .card-color-btn {
    aspect-ratio: 1;
    border-radius: 12px;
    border: 3px solid transparent;
    cursor: pointer;
    transition: all 0.2s ease;
    position: relative;
}

.card-color-btn:hover {
    transform: scale(1.05);
    border-color: rgba(0, 0, 0, 0.2);
}

.dark .card-color-btn:hover {
    border-color: rgba(255, 255, 255, 0.2);
}

.color-check {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    color: white;
    font-size: 24px;
    font-weight: bold;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.3);
}
    
</style>

<script>
    const cardColors = {
    green: 'linear-gradient(135deg, #059669 0%, #047857 100%)',
    blue: 'linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%)',
    purple: 'linear-gradient(135deg, #9333ea 0%, #7e22ce 100%)',
    red: 'linear-gradient(135deg, #dc2626 0%, #b91c1c 100%)',
    orange: 'linear-gradient(135deg, #ea580c 0%, #c2410c 100%)',
    pink: 'linear-gradient(135deg, #ec4899 0%, #db2777 100%)'
};

function changeCardColor(color) {
    document.querySelectorAll('.color-check').forEach(check => check.textContent = '');
    document.getElementById('check-' + color).textContent = '✓';
    
    const previewCard = document.getElementById('preview-card');
    previewCard.style.background = cardColors[color];
    
    document.querySelectorAll('.wallet-card').forEach(card => {
        if (!card.classList.contains('wallet-card-disabled')) {
            card.style.background = cardColors[color];
        }
    });
    
    localStorage.setItem('vert_chasseur_card_color', color);
}

document.addEventListener('DOMContentLoaded', function() {
    const savedColor = localStorage.getItem('vert_chasseur_card_color') || 'green';
    changeCardColor(savedColor);
});
</script>
<style>
    .tabs-dropdown {
    position: relative;
    margin-bottom: 32px;
}

.tabs-dropdown-trigger {
    width: 100%;
    padding: 16px 20px;
    background: rgba(0, 0, 0, 0.05);
    border: 1px solid rgba(0, 0, 0, 0.1);
    border-radius: 12px;
    color: #000;
    font-size: 16px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.dark .tabs-dropdown-trigger {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
    color: #fff;
}

.tabs-dropdown-trigger:hover {
    background: rgba(0, 0, 0, 0.08);
}

.dark .tabs-dropdown-trigger:hover {
    background: rgba(255, 255, 255, 0.08);
}

.dropdown-arrow {
    width: 20px;
    height: 20px;
    transition: transform 0.3s ease;
}

.tabs-dropdown-trigger.open .dropdown-arrow {
    transform: rotate(180deg);
}

.tabs-dropdown-menu {
    position: absolute;
    top: calc(100% + 8px);
    left: 0;
    right: 0;
    background: white;
    border: 1px solid rgba(0, 0, 0, 0.1);
    border-radius: 12px;
    box-shadow: 0 8px 24px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    opacity: 0;
    visibility: hidden;
    transform: translateY(-10px);
    transition: all 0.3s ease;
    z-index: 100;
    max-height: 400px;
    overflow-y: auto;
}

.dark .tabs-dropdown-menu {
    background: #000;
    border: 1px solid rgba(255, 255, 255, 0.1);
    box-shadow: 0 8px 24px rgba(0, 0, 0, 0.3);
}

.tabs-dropdown-menu.show {
    opacity: 1;
    visibility: visible;
    transform: translateY(0);
}

.tabs-dropdown-item {
    width: 100%;
    padding: 14px 20px;
    background: transparent;
    border: none;
    border-bottom: 1px solid rgba(0, 0, 0, 0.05);
    color: #000;
    font-size: 15px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    text-align: left;
}

.dark .tabs-dropdown-item {
    border-bottom: 1px solid rgba(255, 255, 255, 0.05);
    color: #fff;
}

.tabs-dropdown-item:last-child {
    border-bottom: none;
}

.tabs-dropdown-item:hover {
    background: rgba(0, 0, 0, 0.05);
}

.dark .tabs-dropdown-item:hover {
    background: rgba(255, 255, 255, 0.05);
}

.tabs-dropdown-item.active {
    background: rgba(22, 163, 74, 0.1);
    color: #16a34a;
}

.dark .tabs-dropdown-item.active {
    background: rgba(22, 163, 74, 0.15);
}
</style>
<script>
    function toggleTabsMenu() {
    const menu = document.getElementById('tabsDropdownMenu');
    const trigger = document.querySelector('.tabs-dropdown-trigger');
    
    menu.classList.toggle('show');
    trigger.classList.toggle('open');
}

function switchTabFromDropdown(event, tabName, displayName) {
    const tabButtons = document.querySelectorAll('.tabs-dropdown-item');
    const tabContents = document.querySelectorAll('.tab-content');
    
    tabButtons.forEach(btn => btn.classList.remove('active'));
    tabContents.forEach(content => content.classList.remove('active'));
    
    event.target.classList.add('active');
    
    setTimeout(() => {
        document.getElementById(tabName).classList.add('active');
    }, 50);
    
    document.getElementById('current-tab-name').textContent = displayName;
    
    toggleTabsMenu();
}

document.addEventListener('click', function(event) {
    const dropdown = document.querySelector('.tabs-dropdown');
    const menu = document.getElementById('tabsDropdownMenu');
    const trigger = document.querySelector('.tabs-dropdown-trigger');
    
    if (dropdown && !dropdown.contains(event.target)) {
        menu.classList.remove('show');
        trigger.classList.remove('open');
    }
});
</script>

<div id="settingsOverlay" class="overlay">
   <div class="overlay-content">
       <div class="max-w-2xl mx-auto">
        <div class="overlay-header">
            <div>
                <h1 class="overlay-title">Compte</h1>
                <p class="overlay-description">Personnalise ton expérience<br> dans l'application selon tes préférences</p>
            </div>
            <button onclick="closeSettings()" class="overlay-close">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                </svg>
            </button>
        </div>

           <div class="tabs-dropdown">
            <button class="tabs-dropdown-trigger" onclick="toggleTabsMenu()">
                <span id="current-tab-name">Détails de la carte</span>
                <svg class="dropdown-arrow" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"/>
                </svg>
            </button>
            
            <div class="tabs-dropdown-menu" id="tabsDropdownMenu">
                <button class="tabs-dropdown-item active" onclick="switchTabFromDropdown(event, 'wallet', 'Détails de la carte')">Détails de la carte</button>
                <button class="tabs-dropdown-item" onclick="switchTabFromDropdown(event, 'card-custom', 'Personnalisation')">Personnalisation</button>
                <button class="tabs-dropdown-item" onclick="switchTabFromDropdown(event, 'profile', 'Profil')">Profil</button>
                <button class="tabs-dropdown-item" onclick="switchTabFromDropdown(event, 'security', 'Sécurité')">Sécurité</button>
                <button class="tabs-dropdown-item" onclick="switchTabFromDropdown(event, 'licence', 'Licence')">Licence</button>
                <button class="tabs-dropdown-item" onclick="switchTabFromDropdown(event, 'account', 'Informations du compte')">Informations du compte</button>
            </div>
        </div>

           <?php if ($success_message): ?>
           <div class="alert alert-success">
               <?= htmlspecialchars($success_message) ?>
           </div>
           <?php endif; ?>

           <?php if ($error_message): ?>
           <div class="alert alert-error">
               <?= htmlspecialchars($error_message) ?>
           </div>
           <?php endif; ?>

           <div id="wallet" class="tab-content active">
               <?php if ($is_premium): ?>
               <div class="wallet-card">
                    <?= $badge ?>
                    
                    <div class="wallet-content">
                        <div class="wallet-header">
                            <h3 class="wallet-name"><?= htmlspecialchars($resident['first_name'] . ' ' . $resident['last_name']) ?></h3>
                            <p class="wallet-title">Membre Vert Chasseur</p>
                        </div>
                        
                        <div class="wallet-qr-center">
                            <div class="qr-container-center">
                                <img src="https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=<?= urlencode($resident['id']) ?>" 
                                     alt="QR Code" 
                                     class="qr-img-center">
                            </div>
                        </div>
                        
                        <div class="wallet-footer">
                            <div class="wallet-id">ID: <?= $resident['id'] ?></div>
                            <div class="wallet-address"><?= $address ?></div>
                        </div>
                    </div>
                </div>

               <div class="info-card">
                   <h3>Gestion de l'abonnement</h3>
                   <p class="text-sm text-black dark:text-white mb-4">Modifie ton abonnement, consulte tes factures ou annule ton abonnement.</p>
                   <button type="button" onclick="manageSubscription()" class="btn-primary">
                       Gérer mon abonnement
                   </button>
               </div>
               
               <div class="info-card">
                   <div class="flex items-center justify-between mb-4">
                       <h3 class="text-black dark:text-white">Commerces partenaires</h3>
                   </div>
                   <div class="mb-4 p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-800">
                       <p class="text-sm text-blue-800 dark:text-blue-200">Les commerces marqués en jaune seront bientôt disponibles mais sont en cours d'ajout. Ceux en vert acceptent déjà ta carte membre.</p>
                   </div>
                   <div>
                       <div class="space-y-3">
                           <div class="p-3 bg-green-50 dark:bg-green-900/20 rounded-lg border border-green-200 dark:border-green-800">
                               <div class="flex items-center space-x-2">
                                   <span class="w-3 h-3 bg-green-500 rounded-full"></span>
                                   <span class="text-sm font-semibold text-black dark:text-white">Le Delitraiteur</span>
                               </div>
                           </div>
                           
                           <div class="space-y-2">
                               <div class="p-3 bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border border-yellow-200 dark:border-yellow-800">
                                   <div class="flex items-center space-x-2">
                                       <span class="w-3 h-3 bg-yellow-500 rounded-full"></span>
                                       <span class="text-sm text-black dark:text-white">Carrefour Express (en attente)</span>
                                   </div>
                               </div>
                               <div class="p-3 bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border border-yellow-200 dark:border-yellow-800">
                                   <div class="flex items-center space-x-2">
                                       <span class="w-3 h-3 bg-yellow-500 rounded-full"></span>
                                       <span class="text-sm text-black dark:text-white">La Villa Lorraine Traiteur (en attente)</span>
                                   </div>
                               </div>
                               <div class="p-3 bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border border-yellow-200 dark:border-yellow-800">
                                   <div class="flex items-center space-x-2">
                                       <span class="w-3 h-3 bg-yellow-500 rounded-full"></span>
                                       <span class="text-sm text-black dark:text-white">Mille Restaurante (en attente)</span>
                                   </div>
                               </div>
                           </div>
                       </div>
                   </div>
               </div>
               
               <div class="info-card">
                   <div class="flex items-center justify-between mb-4">
                       <h3 class="text-black dark:text-white">Comment utiliser ta carte ?</h3>
                   </div>
                   <div>
                       <div class="space-y-3 text-sm text-black dark:text-white">
                           <div class="flex items-start space-x-3">
                               <span class="flex-shrink-0 w-6 h-6 bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300 rounded-full flex items-center justify-center text-xs font-bold">1</span>
                               <p>Présente ta carte chez les commerçants partenaires de Vert Chasseur</p>
                           </div>
                           <div class="flex items-start space-x-3">
                               <span class="flex-shrink-0 w-6 h-6 bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300 rounded-full flex items-center justify-center text-xs font-bold">2</span>
                               <p>Montre le code-barres au commerçant pour scanner ta carte</p>
                           </div>
                           <div class="flex items-start space-x-3">
                               <span class="flex-shrink-0 w-6 h-6 bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300 rounded-full flex items-center justify-center text-xs font-bold">3</span>
                               <p>Profite des réductions de 10% chez nos partenaires</p>
                           </div>
                       </div>
                       
                       <div class="mt-6 pt-4 border-t border-stone-200 dark:border-stone-700">
                           <h4 class="font-semibold text-black dark:text-white mb-3">Tes avantages membre :</h4>
                           <div class="space-y-2">
                               <div class="flex items-center space-x-2">
                                   <span class="w-2 h-2 bg-green-500 rounded-full"></span>
                                   <span class="text-sm text-black dark:text-white">Réductions de 10% chez nos partenaires</span>
                               </div>
                               <div class="flex items-center space-x-2">
                                   <span class="w-2 h-2 bg-green-500 rounded-full"></span>
                                   <span class="text-sm text-black dark:text-white">Accès à tous les services du quartier</span>
                               </div>
                           </div>
                       </div>
                   </div>
               </div>
               
               <div class="info-card">
                   <h3>Informations de la carte</h3>
                   <div class="info-item">
                       <span class="info-label">Statut</span>
                       <span class="info-value">Membre Premium</span>
                   </div>
                   <div class="info-item">
                       <span class="info-label">Type</span>
                       <span class="info-value">Résident Vert Chasseur</span>
                   </div>
                   <div class="info-item">
                       <span class="info-label">Réductions</span>
                       <span class="info-value">10% chez les partenaires</span>
                   </div>
                   <div class="info-item">
                       <span class="info-label">Membre depuis</span>
                       <span class="info-value"><?= date('d/m/Y', strtotime($resident['created_at'])) ?></span>
                   </div>
                   <?php if ($subscription_months > 0): ?>
                   <div class="info-item">
                       <span class="info-label">Abonné depuis</span>
                       <span class="info-value"><?= $subscription_months ?> mois</span>
                   </div>
                   <?php endif; ?>
               </div>
               <?php else: ?>
               <div class="wallet-card wallet-card-disabled">
                   <div class="wallet-content">
                       <div class="subscription-prompt">
                           <svg class="w-16 h-16 mx-auto mb-4 opacity-80" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                               <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"/>
                           </svg>
                           <h3>Obtiens la carte membre</h3>
                           <p>Abonne-toi pour débloquer ta carte membre et profiter des avantages exclusifs</p>
                           <div class="subscription-price">1,99€</div>
                           <div class="subscription-period">par mois</div>
                           <button type="button" onclick="subscribeNow()" class="btn-subscribe">
                               S'abonner maintenant
                           </button>
                           <button type="button" onclick="refreshMembership()" class="btn-refresh">
                               <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                   <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/>
                               </svg>
                               Rafraîchir le statut
                           </button>
                       </div>
                   </div>
               </div>
               
               <div class="info-card">
                   <h3>Informations importantes</h3>
                   <div class="p-4 bg-amber-50 dark:bg-amber-900/20 rounded-lg border border-amber-200 dark:border-amber-800 mb-4">
                       <div class="flex items-start space-x-3">
                           <svg class="w-5 h-5 text-amber-600 dark:text-amber-400 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                               <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                           </svg>
                           <div class="text-sm text-amber-800 dark:text-amber-200">
                               <p class="font-semibold mb-2">Programme en développement</p>
                               <p>Le programme membre Vert Chasseur est actuellement en cours d'implémentation. Nous travaillons activement à intégrer de nouveaux commerces dans le quartier. Patience, de nombreuses opportunités arrivent bientôt !</p>
                           </div>
                       </div>
                   </div>
                   <p class="text-sm text-stone-600 dark:text-stone-400">Tu seras notifié dès que de nouveaux partenaires rejoindront le programme.</p>
               </div>
               
               <div class="info-card">
                   <h3>Avantages de l'abonnement</h3>
                   <div class="space-y-3 text-sm text-black dark:text-white">
                       <div class="flex items-start space-x-3">
                           <svg class="w-5 h-5 text-green-600 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                               <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                           </svg>
                           <p>Carte de membre digitale avec code-barres</p>
                       </div>
                       <div class="flex items-start space-x-3">
                           <svg class="w-5 h-5 text-green-600 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                               <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                           </svg>
                           <p>10% de réduction chez tous les partenaires</p>
                       </div>
                       <div class="flex items-start space-x-3">
                           <svg class="w-5 h-5 text-green-600 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                               <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                           </svg>
                           <p>Accès prioritaire aux offres spéciales</p>
                       </div>
                       <div class="flex items-start space-x-3">
                           <svg class="w-5 h-5 text-green-600 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                               <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                           </svg>
                           <p>Support de l'économie locale du quartier</p>
                       </div>
                   </div>
               </div>
               <?php endif; ?>
           </div>
           
           <div id="card-custom" class="tab-content">
    <div class="info-card">
        <h3>Personnalise ta carte membre</h3>
        <p class="text-sm text-stone-600 dark:text-stone-400 mb-6">Choisis la couleur de ta carte pour la rendre unique</p>
        
        <div class="grid grid-cols-3 gap-3 mb-6">
    <button onclick="changeCardColor('green')" class="card-color-btn" style="background: linear-gradient(135deg, #059669 0%, #047857 100%);">
        <span class="color-check" id="check-green">✓</span>
    </button>
    <button onclick="changeCardColor('forest')" class="card-color-btn" style="background: linear-gradient(135deg, #065f46 0%, #064e3b 100%);">
        <span class="color-check" id="check-forest"></span>
    </button>
    <button onclick="changeCardColor('sage')" class="card-color-btn" style="background: linear-gradient(135deg, #6b9080 0%, #4f6f5c 100%);">
        <span class="color-check" id="check-sage"></span>
    </button>
    <button onclick="changeCardColor('mint')" class="card-color-btn" style="background: linear-gradient(135deg, #a7c4bc 0%, #84a59d 100%);">
        <span class="color-check" id="check-mint"></span>
    </button>
    <button onclick="changeCardColor('olive')" class="card-color-btn" style="background: linear-gradient(135deg, #8a9a5b 0%, #6b7c3d 100%);">
        <span class="color-check" id="check-olive"></span>
    </button>
    <button onclick="changeCardColor('pine')" class="card-color-btn" style="background: linear-gradient(135deg, #2d5016 0%, #1e3a0f 100%);">
        <span class="color-check" id="check-pine"></span>
    </button>
</div>
        
        <p class="text-sm text-stone-600 dark:text-stone-400 mb-4">Aperçu :</p>
        
        <div class="wallet-card" id="preview-card">
            <?= $badge ?>
            <div class="wallet-content">
                <div class="wallet-header">
                    <h3 class="wallet-name"><?= htmlspecialchars($resident['first_name'] . ' ' . $resident['last_name']) ?></h3>
                    <p class="wallet-title">Membre Vert Chasseur</p>
                </div>
                <div class="wallet-qr-center">
                    <div class="qr-container-center">
                        <img src="https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=<?= urlencode($resident['id']) ?>" alt="QR Code" class="qr-img-center">
                    </div>
                </div>
                <div class="wallet-footer">
                    <div class="wallet-id">ID: <?= $resident['id'] ?></div>
                    <div class="wallet-address"><?= $address ?></div>
                </div>
            </div>
        </div>
    </div>
</div>

           <div id="profile" class="tab-content">
               <form method="POST">
                   <input type="hidden" name="action" value="update_profile">
                   
                   <div class="form-group">
                       <label class="form-label">Prénom *</label>
                       <input type="text" name="first_name" value="<?= htmlspecialchars($resident['first_name']) ?>" required class="form-input">
                   </div>
                   
                   <div class="form-group">
                       <label class="form-label">Nom *</label>
                       <input type="text" name="last_name" value="<?= htmlspecialchars($resident['last_name']) ?>" required class="form-input">
                   </div>
                   
                   <div class="form-group">
                       <label class="form-label">Email *</label>
                       <input type="email" name="email" value="<?= htmlspecialchars($resident['email']) ?>" required class="form-input">
                   </div>
                   
                   <div class="form-group">
                       <label class="form-label">Adresse</label>
                       <textarea name="address" rows="3" class="form-input form-textarea"><?= htmlspecialchars($resident['address']) ?></textarea>
                   </div>
                   
                   <button type="submit" class="btn-primary">Mettre à jour le profil</button>
               </form>
           </div>

           <div id="security" class="tab-content">
               <form method="POST">
                   <input type="hidden" name="action" value="change_password">
                   
                   <div class="form-group">
                       <label class="form-label">Mot de passe actuel *</label>
                       <input type="password" name="current_password" required class="form-input">
                   </div>
                   
                   <div class="form-group">
                       <label class="form-label">Nouveau mot de passe *</label>
                       <input type="password" name="new_password" required minlength="6" class="form-input">
                   </div>
                   
                   <div class="form-group">
                       <label class="form-label">Confirmer le nouveau mot de passe *</label>
                       <input type="password" name="confirm_password" required minlength="6" class="form-input">
                   </div>
                   
                   <button type="submit" class="btn-primary">Changer le mot de passe</button>
               </form>
           </div>

           <div id="licence" class="tab-content">
               <div class="info-card">
                   <h3>Mentions légales</h3>
                   <div class="space-y-4 text-sm text-black dark:text-white">
                       <div>
                           <h4 class="font-semibold mb-2">Éditeur de l'application</h4>
                           <p>Izhak Aktas</p>
                       </div>
                       
                       
                       <div>
                           <h4 class="font-semibold mb-2">Propriété intellectuelle</h4>
                           <p>L'ensemble du contenu de cette application (textes, images, logos, icônes) est la propriété exclusive du propriétaire (Izhak Aktas). Toute reproduction sans autorisation est interdite.</p>
                       </div>
                       
                       <div>
                           <h4 class="font-semibold mb-2">Données personnelles</h4>
                           <p>Conformément au RGPD, tu disposes d'un droit d'accès, de rectification et de suppression de tes données personnelles. Pour exercer ces droits, contacte-nous via les paramètres de ton compte.</p>
                       </div>
                       
                       <div>
                           <h4 class="font-semibold mb-2">Conditions d'utilisation</h4>
                           <p>L'utilisation de cette application implique l'acceptation des présentes mentions légales. Les informations fournies sont données à titre indicatif et ne sauraient engager notre responsabilité.</p>
                       </div>
                       
                       <div>
                           <h4 class="font-semibold mb-2">Contact</h4>
                           <p>Pour toute question concernant l'application ou tes données personnelles, tu peux nous contacter via la section Compte.</p>
                       </div>

                       <div>
                           <h4 class="font-semibold mb-2">Pages légales</h4>
                           <div class="space-y-2">
                               <a href="https://vertchasseur.com/terms-of-use" target="_blank" class="text-green-600 dark:text-green-400 hover:underline">Consulter les condition d'utilisations</a>
                               <br>
                               <a href="https://vertchasseur.com/privacy" target="_blank" class="text-green-600 dark:text-green-400 hover:underline">Consulter la politique de confidentialité</a>
                           </div>
                       </div>
                   </div>
               </div>
               
               <div class="info-card">
                   <h3>Licence de l'application</h3>
                   <div class="text-sm text-black dark:text-white space-y-3">
                       <p>Version 1.5.0 - © 2025 Vert Chasseur</p>
                       <p>Cette application est fournie "en l'état" sans garantie d'aucune sorte. L'utilisation de cette application est soumise aux conditions générales d'utilisation disponibles sur demande.</p>
                       <p>Technologie développée par Imators LLC</p>
                   </div>
               </div>
           </div>

           <div id="account" class="tab-content">               
               <div class="info-card">
                   <h3>Informations du compte</h3>
                   <div class="info-item">
                       <span class="info-label">ID</span>
                       <span class="info-value"><?= $resident['id'] ?></span>
                   </div>
                   <div class="info-item">
                       <span class="info-label">Statut</span>
                       <span class="info-value"><?= $is_premium ? 'Membre Premium' : 'Membre Standard' ?></span>
                   </div>
                   <div class="info-item">
                       <span class="info-label">Membre depuis</span>
                       <span class="info-value"><?= date('d/m/Y à H:i', strtotime($resident['created_at'])) ?></span>
                   </div>
                   <div class="info-item">
                       <span class="info-label">Dernière connexion</span>
                       <span class="info-value"><?= isset($resident['last_login']) ? date('d/m/Y à H:i', strtotime($resident['last_login'])) : 'Jamais' ?></span>
                   </div>
               </div>
               
               <div class="danger-zone">
                   <h3>Zone de danger</h3>
                   <p>Actions irréversibles concernant ton compte.</p>
                   
                   <a href="/logout" onclick="return confirm('Es-tu sûr de vouloir te déconnecter ?')" class="btn-danger">
                       <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                           <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"/>
                       </svg>
                       Se déconnecter
                   </a>
                   <a href="delete-account.php" class="btn-danger">
                       <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                      </svg>
                      Supprimer mon compte
                  </a>
              </div>
          </div>
      </div>
  </div>
</div>
<?php endif; ?>

<script>
let poemExpanded = false;
let weatherExpanded = false;
let weatherRefreshInterval = null;

function togglePoem() {
    const content = document.getElementById('poem-content');
    const gradient = document.getElementById('poem-gradient');
    const toggleText = document.getElementById('poem-toggle-text');
    const toggleIcon = document.getElementById('poem-toggle-icon');
    
    if (!poemExpanded) {
        content.style.maxHeight = content.scrollHeight + 'px';
        gradient.style.opacity = '0';
        toggleText.textContent = 'Voir moins';
        toggleIcon.style.transform = 'rotate(180deg)';
        poemExpanded = true;
    } else {
        content.style.maxHeight = '200px';
        gradient.style.opacity = '1';
        toggleText.textContent = 'Voir plus';
        toggleIcon.style.transform = 'rotate(0deg)';
        poemExpanded = false;
    }
}

function toggleWeatherExpansion() {
    const expandedDiv = document.getElementById('weather-expanded');
    
    if (!weatherExpanded) {
        loadHourlyWeather();
        loadForecast();
        expandedDiv.classList.add('show');
        weatherExpanded = true;
    } else {
        expandedDiv.classList.remove('show');
        weatherExpanded = false;
    }
}

async function loadWeather() {
    try {
        const response = await fetch('https://api.weatherapi.com/v1/current.json?key=fa1a36a37c3f4e7ab7e233043252305&q=Uccle,Belgium&lang=fr');
        
        if (!response.ok) {
            throw new Error('API non disponible');
        }
        
        const data = await response.json();
        
        const weatherCurrent = document.getElementById('weather-current');
        const weatherSummary = document.getElementById('weather-summary');
        
        const iconUrl = `https:${data.current.condition.icon}`;
        
        weatherCurrent.style.opacity = '0';
        weatherCurrent.style.transition = 'opacity 0.3s ease';
        
        setTimeout(() => {
            weatherCurrent.innerHTML = `
                <div class="flex items-center space-x-4">
                    <img src="${iconUrl}" alt="${data.current.condition.text}" class="w-12 h-12 md:w-16 md:h-16">
                    <div>
                        <div class="text-2xl md:text-3xl font-bold">${Math.round(data.current.temp_c)}°C</div>
                        <div class="text-sm md:text-base opacity-90">${data.current.condition.text}</div>
                    </div>
                </div>
            `;
            weatherCurrent.style.opacity = '1';
        }, 150);
        
        weatherSummary.style.opacity = '0';
        weatherSummary.style.transition = 'opacity 0.3s ease';
        
        setTimeout(() => {
            weatherSummary.innerHTML = `
                <div class="text-xs md:text-sm space-y-1">
                    <div>Ressenti: ${Math.round(data.current.feelslike_c)}°C</div>
                    <div>Humidité: ${data.current.humidity}%</div>
                    <div>Vent: ${Math.round(data.current.wind_kph)} km/h</div>
                </div>
            `;
            weatherSummary.style.opacity = '1';
        }, 150);
        
    } catch (error) {
        const weatherCurrent = document.getElementById('weather-current');
        const weatherSummary = document.getElementById('weather-summary');
        
        weatherCurrent.innerHTML = `
            <div class="flex items-center space-x-4">
                <div class="w-12 h-12 md:w-16 md:h-16 bg-white/20 rounded-lg flex items-center justify-center">
                    <svg class="w-6 h-6 md:w-8 md:h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 15a4 4 0 004 4h9a5 5 0 10-.1-9.999 5.002 5.002 0 10-9.78 2.096A4.001 4.001 0 003 15z"/>
                    </svg>
                </div>
                <div>
                    <div class="text-2xl md:text-3xl font-bold">--°C</div>
                    <div class="text-sm md:text-base opacity-90">Météo indisponible</div>
                </div>
            </div>
        `;
        
        weatherSummary.innerHTML = `
            <div class="text-xs md:text-sm opacity-75">
                Service temporairement indisponible
            </div>
        `;
    }
}

async function loadHourlyWeather() {
    try {
        const response = await fetch('https://api.weatherapi.com/v1/forecast.json?key=fa1a36a37c3f4e7ab7e233043252305&q=Uccle,Belgium&hours=24&lang=fr');
        
        if (!response.ok) {
            throw new Error('API non disponible');
        }
        
        const data = await response.json();
        const hourlyContainer = document.getElementById('hourly-forecast');
        
        let hourlyHtml = '';
        const currentHour = new Date().getHours();
        
        for (let i = 0; i < 12; i++) {
            const hour = data.forecast.forecastday[0].hour[(currentHour + i) % 24];
            const hourTime = new Date(hour.time);
            const displayHour = i === 0 ? 'Now' : hourTime.getHours() + 'h';
            
            hourlyHtml += `
                <div class="hour-item">
                    <div class="text-xs font-medium mb-2">${displayHour}</div>
                    <img src="https:${hour.condition.icon}" alt="${hour.condition.text}" class="w-8 h-8 mx-auto mb-2">
                    <div class="text-xs font-bold">${Math.round(hour.temp_c)}°</div>
                </div>
            `;
        }
        
        hourlyContainer.innerHTML = hourlyHtml;
        
    } catch (error) {
        const hourlyContainer = document.getElementById('hourly-forecast');
        hourlyContainer.innerHTML = `
            <div class="text-center py-4 opacity-75 w-full">
                <p class="text-xs">Prévisions horaires indisponibles</p>
            </div>
        `;
    }
}

async function loadForecast() {
    try {
        const response = await fetch('https://api.weatherapi.com/v1/forecast.json?key=fa1a36a37c3f4e7ab7e233043252305&q=Uccle,Belgium&days=3&lang=fr');
        
        if (!response.ok) {
            throw new Error('API non disponible');
        }
        
        const data = await response.json();
        const forecastContainer = document.getElementById('forecast-days');
        
        let forecastHtml = '';
        
        data.forecast.forecastday.forEach((day, index) => {
            const date = new Date(day.date);
            const dayName = index === 0 ? 'Aujourd\'hui' : 
                           index === 1 ? 'Demain' : 
                           date.toLocaleDateString('fr-FR', { weekday: 'long' });
            
            forecastHtml += `
                <div class="forecast-day">
                    <div class="flex items-center space-x-4">
                        <img src="https:${day.day.condition.icon}" alt="${day.day.condition.text}" class="w-10 h-10">
                        <div class="flex-1">
                            <div class="font-medium text-sm">${dayName}</div>
                            <div class="text-xs opacity-75">${day.day.condition.text}</div>
                        </div>
                    </div>
                    <div class="text-right">
                        <div class="font-bold text-sm">${Math.round(day.day.maxtemp_c)}° / ${Math.round(day.day.mintemp_c)}°</div>
                        <div class="text-xs opacity-75">${Math.round(day.day.avghumidity)}% • ${Math.round(day.day.maxwind_kph)} km/h</div>
                    </div>
                </div>
            `;
        });
        
        forecastContainer.innerHTML = forecastHtml;
        
    } catch (error) {
        const forecastContainer = document.getElementById('forecast-days');
        forecastContainer.innerHTML = `
            <div class="text-center py-4 opacity-75">
                <p>Prévisions indisponibles</p>
            </div>
        `;
    }
}

<?php if (!$is_guest): ?>
const shopsData = <?= $shops_json ?>;
let currentSelectedShop = null;

function getDistance(lat1, lon1, lat2, lon2) {
    const R = 6371;
    const dLat = (lat2 - lat1) * Math.PI / 180;
    const dLon = (lon2 - lon1) * Math.PI / 180;
    const a = Math.sin(dLat/2) * Math.sin(dLat/2) + Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) * Math.sin(dLon/2) * Math.sin(dLon/2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
    return R * c * 1000;
}

function openCardMember() {
    if (!navigator.geolocation) {
        alert('La géolocalisation n\'est pas disponible sur ce navigateur.');
        return;
    }

    navigator.geolocation.getCurrentPosition(
        function(position) {
            const userLat = position.coords.latitude;
            const userLon = position.coords.longitude;
            
            let nearbyShops = [];
            
            shopsData.forEach(shop => {
                const distance = getDistance(userLat, userLon, parseFloat(shop.latitude), parseFloat(shop.longitude));
                if (distance <= 50) {
                    nearbyShops.push({
                        ...shop,
                        distance: distance
                    });
                }
            });
            
            if (nearbyShops.length > 0) {
                const closestShop = nearbyShops.reduce((prev, current) => 
                    prev.distance < current.distance ? prev : current
                );
                
                currentSelectedShop = closestShop;
                    document.getElementById('cardDialogMessage').textContent = 
                        `Il semblerais que vous êtes chez ${closestShop.title}. Souhaitez-vous utiliser votre carte pour ce commerce ?`;
                
                document.getElementById('cardOverlayBlur').classList.add('show');
                document.querySelector('.card-dialog-btn-yes').style.display = 'flex';
                document.querySelector('.card-dialog-btn-no').textContent = 'Non';
                document.querySelector('.card-dialog-btn-no').onclick = closeCardDialog;
            } else {
                document.getElementById('cardDialogMessage').textContent = 
                    'Vous semblez être à côté d\'aucun commerces pour utiliser votre carte.';
                
                document.getElementById('cardOverlayBlur').classList.add('show');
                document.querySelector('.card-dialog-btn-yes').style.display = 'none';
                document.querySelector('.card-dialog-btn-no').textContent = 'Voir ma carte';
                document.querySelector('.card-dialog-btn-no').onclick = function() {
                    closeCardDialog();
                    showCardDisplay();
                };
            }
        },
        function(error) {
            alert('Impossible d\'accéder à ta localisation. Vérifie tes permissions.');
        }
    );
}

function closeCardDialog() {
    document.getElementById('cardOverlayBlur').classList.remove('show');
}

async function confirmCardUsage() {
    if (currentSelectedShop) {
        try {
            const formData = new FormData();
            formData.append('track_card_usage', '1');
            formData.append('shop_id', currentSelectedShop.id);
            
            const response = await fetch(window.location.href, {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            
            if (data.success) {
                alert('Carte utilisée avec succès! +5 points\nTotal: ' + data.points + ' points');
            } else {
                alert('Erreur: ' + (data.error || 'Erreur inconnue'));
            }
            
        } catch (error) {
            console.error('Erreur:', error);            
        }
    }
    
    closeCardDialog();
    showCardDisplay();
}


function showCardDisplay() {
    document.getElementById('cardDisplayOverlay').classList.add('show');
}

function closeCardDisplay() {
    document.getElementById('cardDisplayOverlay').classList.remove('show');
}

async function subscribeNow() {
    try {
        const response = await fetch('generate-sso-token.php');
        const data = await response.json();
        
        if (data.success) {
            const subscriptionUrl = `https://imators.systems/vertchasseur/subscription.php?sso_token=${data.token}`;
            window.location.href = subscriptionUrl;
        } else {
            alert('Erreur d\'authentification');
        }
    } catch (error) {
        alert('Erreur de connexion');
    }
}

async function manageSubscription() {
    window.location.href = 'manage-subscription.php';
}

async function refreshMembership() {
    const refreshBtn = event.target;
    const originalHtml = refreshBtn.innerHTML;
    
    refreshBtn.disabled = true;
    refreshBtn.innerHTML = `
        <svg class="w-4 h-4 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"/>
        </svg>
        Vérification...
    `;
    
    try {
        await new Promise(resolve => setTimeout(resolve, 1000));
        window.location.reload();
    } catch (error) {
        refreshBtn.innerHTML = originalHtml;
        refreshBtn.disabled = false;
        alert('Erreur lors de la vérification. Réessaye dans quelques instants.');
    }
}

function openHelp() {
    document.getElementById('helpOverlay').classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closeHelp() {
    document.getElementById('helpOverlay').classList.remove('show');
    document.body.style.overflow = 'auto';
}

function openTicketSystem() {
    window.location.href = 'https://www.imators.com/support';
}

function openSettings() {
    document.getElementById('settingsOverlay').classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closeSettings() {
    document.getElementById('settingsOverlay').classList.remove('show');
    document.body.style.overflow = 'auto';
}

function switchTab(event, tabName) {
    const tabButtons = document.querySelectorAll('.tab-btn');
    const tabContents = document.querySelectorAll('.tab-content');
    
    tabButtons.forEach(btn => btn.classList.remove('active'));
    tabContents.forEach(content => {
        content.classList.remove('active');
    });
    
    event.target.classList.add('active');
    
    setTimeout(() => {
        document.getElementById(tabName).classList.add('active');
    }, 50);
}
<?php endif; ?>
</script>

<main class="min-h-screen px-4 py-8 md:py-16">
 <div class="max-w-7xl mx-auto">
<style>
.letter-animate {
    display: inline-block;
    opacity: 0;
    transform: translateY(20px);
    animation: letterUp 0.4s ease-out forwards;
}

@keyframes letterUp {
    to {
        opacity: 1;
        transform: translateY(0);
    }
}
</style>

<div class="flex flex-col items-center justify-center mb-8 md:mb-12 text-center relative">
    <?php if (!$is_guest): ?>
<button onclick="openHelp()" class="absolute top-0 left-0 p-2 text-stone-600 dark:text-stone-400 hover:text-stone-800 dark:hover:text-stone-200 transition-colors" title="Aide">
    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
    </svg>
</button>
    
<button onclick="openSettings()" class="absolute top-0 right-0 p-2 text-stone-600 dark:text-stone-400 hover:text-stone-800 dark:hover:text-stone-200 transition-colors" title="Paramètres">
    <svg class="w-6 h-6" fill="currentColor" viewBox="0 0 24 24">
        <circle cx="12" cy="8" r="4"/>
        <path d="M12 14c-6 0-8 3-8 3v7h16v-7s-2-3-8-3z"/>
    </svg>
</button>
    
    <?php else: ?>
    <a href="/logout" class="absolute top-0 right-0 p-2 text-stone-600 dark:text-stone-400 hover:text-stone-800 dark:hover:text-stone-200 transition-colors" title="Retour">
        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
        </svg>
    </a>
    <?php endif; ?>

    <?php if (!$is_guest): ?>
    <h1 class="mt-4 md:mt-12 text-3xl md:text-5xl lg:text-6xl font-light text-stone-800 dark:text-white mb-2 md:mb-3 tracking-tight">
        Salutations <span id="greeting-text"></span>
    </h1>
    <?php else: ?>
    <h1 class="mt-4 md:mt-12 text-3xl md:text-5xl lg:text-6xl font-light text-stone-800 dark:text-white mb-2 md:mb-3 tracking-tight">
        Bienvenue à <span class="txt-chasseur text-green-600">Vert Chasseur</span>
    </h1>
    <?php endif; ?>
    <p class="text-lg md:text-xl text-stone-600 dark:text-stone-400 font-light">
        Ravis de te voir.
    </p>
</div>

<div class="floating-actions">
<?php if ($is_premium): ?>
    <button id="btnCardMember" class="action-btn active" data-action="card">
        <svg class="action-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 6H5a2 2 0 00-2 2v9a2 2 0 002 2h14a2 2 0 002-2V8a2 2 0 00-2-2h-5m-4 0V4a2 2 0 114 0v2m-4 0a2 2 0 104 0"/>
        </svg>
        <span class="action-text">Carte membre</span>
    </button>
    <?php endif; ?>
    
    <button id="btnFilter" class="action-btn <?php echo !$is_premium ? 'active' : ''; ?>" data-action="filter">
        <svg class="action-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 4a1 1 0 011-1h16a1 1 0 011 1v2.586a1 1 0 01-.293.707l-6.414 6.414a1 1 0 00-.293.707V17l-4 4v-6.586a1 1 0 00-.293-.707L3.293 7.293A1 1 0 013 6.586V4z"/>
        </svg>
        <span class="action-text">Filtres</span>
    </button>
    
    <button id="btnPoints" class="action-btn" data-action="points">
        <svg class="action-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
        </svg>
        <span class="action-text">Points</span>
    </button>
</div>

<div id="filterOverlay" class="overlay">
    <div class="overlay-content-left">
        <div class="max-w-2xl mx-auto">
            <div class="overlay-header">
                <div>
                    <h1 class="overlay-title">Filtres d'affichage</h1>
                    <p class="overlay-description">Personnalise les éléments visibles<br>sur ta page d'accueil</p>
                </div>
                <button onclick="closeFilterOverlay()" class="overlay-close">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                    </svg>
                </button>
            </div>
            
            <div class="info-card">
                <h3>Éléments de la page</h3>
                <p class="text-sm text-stone-600 dark:text-stone-400 mb-4">Décoche les éléments que tu ne souhaites pas voir</p>
                
                <div class="space-y-3">
                    <label class="filter-checkbox-item">
                        <input type="checkbox" id="filterWeather" class="filter-checkbox" checked>
                        <span class="filter-label">Météo</span>
                    </label>
                    
                    <label class="filter-checkbox-item">
                        <input type="checkbox" id="filterPub" class="filter-checkbox" checked>
                        <span class="filter-label">Publicité</span>
                    </label>
                    
                    <label class="filter-checkbox-item">
                        <input type="checkbox" id="filterPoem" class="filter-checkbox" checked>
                        <span class="filter-label">Poème de la semaine</span>
                    </label>
                    
                    <label class="filter-checkbox-item">
                        <input type="checkbox" id="filterOffers" class="filter-checkbox" checked>
                        <span class="filter-label">Offres du moment</span>
                    </label>
                </div>
                
                <button onclick="applyFilters()" class="btn-primary mt-6">Appliquer les filtres</button>
            </div>
        </div>
    </div>
</div>

<style>
.floating-actions {
    display: flex;
    gap: 12px;
    justify-content: flex-start;
    align-items: center;
    margin: 24px 0;
    padding: 0 4px;
}

.action-btn {
    position: relative;
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 14px;
    width: 48px;
    background: rgba(255, 255, 255, 0.7);
    backdrop-filter: blur(12px);
    border: 1px solid rgba(0, 0, 0, 0.1);
    border-radius: 16px;
    cursor: pointer;
    transition: all 0.6s cubic-bezier(0.25, 0.46, 0.45, 0.94);
    box-shadow: 0 4px 16px rgba(0, 0, 0, 0.08);
    white-space: nowrap;
    overflow: hidden;
}

.dark .action-btn {
    background: rgba(0, 0, 0, 0.5);
    border: 1px solid rgba(255, 255, 255, 0.15);
}

.action-btn:hover {
    box-shadow: 0 6px 20px rgba(0, 0, 0, 0.1);
}

.action-btn.active {
    background: rgba(22, 163, 74, 0.15);
    border-color: rgba(22, 163, 74, 0.3);
    padding: 14px 20px;
    width: auto;
}

.dark .action-btn.active {
    background: rgba(22, 163, 74, 0.25);
    border-color: rgba(22, 163, 74, 0.4);
}

.action-icon {
    width: 20px;
    height: 20px;
    color: #000;
    flex-shrink: 0;
    transition: color 0.6s cubic-bezier(0.25, 0.46, 0.45, 0.94);
}

.dark .action-icon {
    color: #fff;
}

.action-btn.active .action-icon {
    color: #16a34a;
}

.action-text {
    font-size: 15px;
    font-weight: 500;
    color: #000;
    opacity: 0;
    width: 0;
    transition: opacity 0.6s cubic-bezier(0.25, 0.46, 0.45, 0.94), width 0.6s cubic-bezier(0.25, 0.46, 0.45, 0.94);
    overflow: hidden;
}

.dark .action-text {
    color: #fff;
}

.action-btn.active .action-text {
    color: #16a34a;
    opacity: 1;
    width: auto;
}

.filter-checkbox-item {
    display: flex;
    align-items: center;
    padding: 16px;
    background: rgba(0, 0, 0, 0.03);
    border: 1px solid rgba(0, 0, 0, 0.1);
    border-radius: 12px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.dark .filter-checkbox-item {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.filter-checkbox-item:hover {
    background: rgba(0, 0, 0, 0.05);
}

.dark .filter-checkbox-item:hover {
    background: rgba(255, 255, 255, 0.08);
}

.filter-checkbox {
    width: 20px;
    height: 20px;
    margin-right: 12px;
    cursor: pointer;
    accent-color: #16a34a;
}

.filter-label {
    font-size: 15px;
    font-weight: 500;
    color: #000;
}

.dark .filter-label {
    color: #fff;
}

@media (max-width: 768px) {
    .floating-actions {
        gap: 8px;
    }
    
    .action-btn {
        padding: 12px;
        width: 44px;
    }
    
    .action-btn.active {
        padding: 12px 16px;
    }
    
    .action-text {
        font-size: 14px;
    }
}
</style>
<script>
let activeAction = 'card';

document.querySelectorAll('.action-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        const action = this.dataset.action;
        
        if (activeAction === action) {
            executeAction(action);
        } else {
            document.querySelectorAll('.action-btn').forEach(b => {
                b.classList.remove('active');
                b.classList.add('collapsed');
            });
            
            this.classList.remove('collapsed');
            this.classList.add('active');
            activeAction = action;
        }
    });
});

function executeAction(action) {
    switch(action) {
        case 'card':
            <?php if ($is_premium): ?>
            openCardMember();
            <?php else: ?>
            alert('Abonne-toi pour accéder à ta carte membre');
            <?php endif; ?>
            break;
        case 'filter':
            openFilterOverlay();
            break;
        case 'points':
            window.location.href = '/points-track';
            break;
    }
}

function openFilterOverlay() {
    loadCurrentFilters();
    document.getElementById('filterOverlay').classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closeFilterOverlay() {
    document.getElementById('filterOverlay').classList.remove('show');
    document.body.style.overflow = 'auto';
}

function loadCurrentFilters() {
    const filters = JSON.parse(localStorage.getItem('homeFilters') || '{}');
    
    document.getElementById('filterWeather').checked = filters.weather !== false;
    document.getElementById('filterPub').checked = filters.pub !== false;
    document.getElementById('filterPoem').checked = filters.poem !== false;
    document.getElementById('filterOffers').checked = filters.offers !== false;
}

function applyFilters() {
    const filters = {
        weather: document.getElementById('filterWeather').checked,
        pub: document.getElementById('filterPub').checked,
        poem: document.getElementById('filterPoem').checked,
        offers: document.getElementById('filterOffers').checked
    };
    
    localStorage.setItem('homeFilters', JSON.stringify(filters));
    
    document.getElementById('weather-section').style.display = filters.weather ? 'block' : 'none';
    document.getElementById('pub-block').style.display = filters.pub ? 'block' : 'none';
    document.getElementById('poem-container').closest('.bg-white').style.display = filters.poem ? 'block' : 'none';
    document.getElementById('offers-section').style.display = filters.offers ? 'block' : 'none';
    
    closeFilterOverlay();
}

document.addEventListener('DOMContentLoaded', function() {
    applyFilters();
});
</script>

<div id="pointsInfoOverlay" class="overlay">
    <div class="overlay-content-left">
        <div class="max-w-2xl mx-auto">
            <div class="overlay-header">
                <div>
                    <h1 class="overlay-title">Système de points</h1>
                    <p class="overlay-description">Gagne des points à chaque utilisation<br>de ta carte membre</p>
                </div>
                <button onclick="closePointsInfo()" class="overlay-close">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                    </svg>
                </button>
            </div>

            <div class="info-card mb-6">
                <h3>Comment fonctionnent les points ?</h3>
                <div class="space-y-3 text-sm text-black dark:text-white mt-4">
                    <div class="flex items-start space-x-3">
                        <span class="flex-shrink-0 w-6 h-6 bg-indigo-100 dark:bg-indigo-800 text-indigo-700 dark:text-indigo-300 rounded-full flex items-center justify-center text-xs font-bold">1</span>
                        <p>Tu gagnes <strong>5 points</strong> à chaque utilisation de ta carte membre</p>
                    </div>
                    <div class="flex items-start space-x-3">
                        <span class="flex-shrink-0 w-6 h-6 bg-indigo-100 dark:bg-indigo-800 text-indigo-700 dark:text-indigo-300 rounded-full flex items-center justify-center text-xs font-bold">2</span>
                        <p>Plus tu accumules de points, plus tu débloques de paliers</p>
                    </div>
                    <div class="flex items-start space-x-3">
                        <span class="flex-shrink-0 w-6 h-6 bg-indigo-100 dark:bg-indigo-800 text-indigo-700 dark:text-indigo-300 rounded-full flex items-center justify-center text-xs font-bold">3</span>
                        <p>Chaque palier peut te donner accès à des récompenses exclusives</p>
                    </div>
                </div>
            </div>

            <div class="info-card mb-6">
                <h3>Les paliers</h3>
                <div class="space-y-3 mt-4">
                    <div class="p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-800">
                        <div class="flex items-center justify-between mb-2">
                            <span class="font-semibold text-blue-900 dark:text-blue-100">Cool</span>
                            <span class="text-xs text-blue-700 dark:text-blue-300">0 - 100 points</span>
                        </div>
                        <p class="text-sm text-blue-800 dark:text-blue-200">Ton premier palier, bienvenue !</p>
                    </div>
                    
                    <div class="p-4 bg-green-50 dark:bg-green-900/20 rounded-lg border border-green-200 dark:border-green-800">
                        <div class="flex items-center justify-between mb-2">
                            <span class="font-semibold text-green-900 dark:text-green-100">Great</span>
                            <span class="text-xs text-green-700 dark:text-green-300">100 - 500 points</span>
                        </div>
                        <p class="text-sm text-green-800 dark:text-green-200"><strong>Récompense :</strong> 10€ offert dans un commerce</p>
                    </div>
                    
                    <div class="p-4 bg-purple-50 dark:bg-purple-900/20 rounded-lg border border-purple-200 dark:border-purple-800">
                        <div class="flex items-center justify-between mb-2">
                            <span class="font-semibold text-purple-900 dark:text-purple-100">Crown</span>
                            <span class="text-xs text-purple-700 dark:text-purple-300">500 - 1000 points</span>
                        </div>
                        <p class="text-sm text-purple-800 dark:text-purple-200"><strong>Récompense :</strong> Panier gourmand 40€</p>
                    </div>
                    
                    <div class="p-4 bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border border-yellow-200 dark:border-yellow-800">
                        <div class="flex items-center justify-between mb-2">
                            <span class="font-semibold text-yellow-900 dark:text-yellow-100">Gold</span>
                            <span class="text-xs text-yellow-700 dark:text-yellow-300">1000 - 5000 points</span>
                        </div>
                        <p class="text-sm text-yellow-800 dark:text-yellow-200">Un statut envié</p>
                    </div>
                    
                    <div class="p-4 bg-orange-50 dark:bg-orange-900/20 rounded-lg border border-orange-200 dark:border-orange-800">
                        <div class="flex items-center justify-between mb-2">
                            <span class="font-semibold text-orange-900 dark:text-orange-100">Hunter</span>
                            <span class="text-xs text-orange-700 dark:text-orange-300">5000+ points</span>
                        </div>
                        <p class="text-sm text-orange-800 dark:text-orange-200">Le plus haut palier, tu es un vrai chasseur !</p>
                    </div>
                </div>
            </div>

            <div class="info-card">
                <h3>Tes récompenses</h3>
                <?php if (mysqli_num_rows($rewards_result) > 0): ?>
                    <div class="space-y-3 mt-4">
                        <?php while ($reward = mysqli_fetch_assoc($rewards_result)): ?>
                        <div class="p-4 bg-indigo-50 dark:bg-indigo-900/20 rounded-lg border border-indigo-200 dark:border-indigo-800">
                            <div class="flex items-start justify-between mb-2">
                                <div>
                                    <span class="font-semibold text-indigo-900 dark:text-indigo-100"><?= htmlspecialchars($reward['tier']) ?></span>
                                    <p class="text-sm text-indigo-700 dark:text-indigo-300 mt-1"><?= htmlspecialchars($reward['reward_description']) ?></p>
                                </div>
                                <span class="text-xs px-2 py-1 rounded-full <?= $reward['status'] === 'claimed' ? 'bg-green-100 dark:bg-green-800 text-green-700 dark:text-green-300' : 'bg-amber-100 dark:bg-amber-800 text-amber-700 dark:text-amber-300' ?>">
                                    <?= $reward['status'] === 'claimed' ? 'Réclamée' : 'En attente' ?>
                                </span>
                            </div>
                            <?php if ($reward['status'] === 'pending'): ?>
                            <button onclick="openHelpAndScroll('wallet')" class="w-full mt-3 bg-indigo-600 hover:bg-indigo-700 text-white text-xs font-medium py-2 px-3 rounded transition-colors">
                                Réclamer cette récompense
                            </button>
                            <?php else: ?>
                            <p class="text-xs text-green-600 dark:text-green-400 mt-3">Réclamée le <?= date('d/m/Y', strtotime($reward['claimed_at'])) ?></p>
                            <?php endif; ?>
                        </div>
                        <?php endwhile; ?>
                    </div>
                <?php else: ?>
                    <p class="text-sm text-stone-600 dark:text-stone-400 mt-4">Aucune récompense débloquée pour le moment. Continue à utiliser ta carte !</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
    function openPointsInfo() {
    document.getElementById('pointsInfoOverlay').classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closePointsInfo() {
    document.getElementById('pointsInfoOverlay').classList.remove('show');
    document.body.style.overflow = 'auto';
}

function openHelpAndScroll(tab) {
    closePointsInfo();
    openHelp();
    setTimeout(() => {
        switchTab({target: {classList: {add: () => {}, remove: () => {}}}}, tab);
    }, 100);
}
</script>

<?php
$alert_active = false;
$alert_title = "Alerte météo pour les prochains jours en cours";
$alert_type = "Températures froides"; 
$alert_start = "17/11/2025";
$alert_end = "23/11/2025";
$alert_description = "Une chute impressionnante de température va avoir lieu durant cette période. Les températures pourrait atteindre jusqu'à -8 degrés de ressenti dans les minima.";
?>

<?php if ($alert_active): ?>
<div class="bg-gradient-to-r from-orange-500 to-red-500 rounded-2xl md:rounded-3xl p-6 md:p-8 shadow-lg mb-6 md:mb-8">
    <div class="flex items-start justify-between">
        <div class="flex items-start space-x-3 flex-1">
            <svg class="w-6 h-6 text-white flex-shrink-0 mt-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
            </svg>
            <div>
                <h3 class="text-white font-semibold text-lg md:text-xl mb-1"><?= $alert_title ?></h3>
                <p class="text-white/90 text-sm">Consultez les détails de cette alerte importante</p>
            </div>
        </div>
        <button onclick="openWeatherAlert()" class="ml-4 bg-white/20 hover:bg-white/30 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors backdrop-blur-sm flex-shrink-0">
            Détails
        </button>
    </div>
</div>
<?php endif; ?>

<div id="weatherAlertOverlay" class="overlay">
    <div class="overlay-content">
        <div class="max-w-2xl mx-auto">
            <div class="overlay-header">
                <div>
                    <h1 class="overlay-title">Alerte météo</h1>
                    <p class="overlay-description">Informations détaillées sur l'alerte en cours</p>
                </div>
                <button onclick="closeWeatherAlert()" class="overlay-close">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                    </svg>
                </button>
            </div>

            <div class="bg-gradient-to-r from-orange-50 to-red-50 dark:from-orange-900/20 dark:to-red-900/20 rounded-2xl p-6 border border-orange-200 dark:border-orange-800 mb-6">
                <div class="flex items-start space-x-3 mb-4">
                    <svg class="w-8 h-8 text-orange-600 dark:text-orange-400 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                    </svg>
                    <div>
                        <h3 class="text-xl font-semibold text-orange-900 dark:text-orange-100"><?= $alert_type ?></h3>
                        <p class="text-sm text-orange-700 dark:text-orange-300 mt-1">Type d'alerte</p>
                    </div>
                </div>
                
                <div class="space-y-4">
                    <div>
                        <label class="text-sm font-medium text-orange-800 dark:text-orange-200 block mb-1">Période</label>
                        <p class="text-orange-900 dark:text-orange-100">Du <?= $alert_start ?> au <?= $alert_end ?></p>
                    </div>
                    
                    <div>
                        <label class="text-sm font-medium text-orange-800 dark:text-orange-200 block mb-2">Description</label>
                        <p class="text-orange-900 dark:text-orange-100 leading-relaxed"><?= $alert_description ?></p>
                    </div>
                </div>
            </div>

            <div class="info-card">
                <h3>Recommandations</h3>
                <div class="space-y-3 text-sm text-black dark:text-white">
                    <div class="flex items-start space-x-3">
                        <svg class="w-5 h-5 text-orange-600 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                        </svg>
                        <p>Couvrez vous de manière suffisante</p>
                    </div>
                    <div class="flex items-start space-x-3">
                        <svg class="w-5 h-5 text-orange-600 flex-shrink-0 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                        </svg>
                        <p>Éviter tout déplacement en vélo (le risque de verglas est présent)</p>
                    </div>                  
                </div>
            </div>

            <div class="info-card">
                <p class="text-sm text-stone-600 dark:text-stone-400 text-center">
                    Source : Institut Royal Météorologique (IRM)
                </p>
            </div>
        </div>
    </div>
</div>

<script>
function openWeatherAlert() {
    document.getElementById('weatherAlertOverlay').classList.add('show');
    document.body.style.overflow = 'hidden';
}

function closeWeatherAlert() {
    document.getElementById('weatherAlertOverlay').classList.remove('show');
    document.body.style.overflow = 'auto';
}
</script>

     <div class="grid grid-cols-1 lg:grid-cols-4 gap-6 md:gap-8 mb-8 md:mb-16">
         <div class="lg:col-span-3 space-y-6 md:space-y-8">
             <div id="weather-section" class="weather-banner" onclick="toggleWeatherExpansion()">
                 <div class="weather-banner-content">
                     <div class="flex justify-between items-start mb-4">
                         <div class="flex-1">
                             <h3 class="text-xl md:text-2xl font-medium mb-3">Météo à Vert Chasseur</h3>
                             <div id="weather-current" class="flex items-center space-x-4">
                                 <div class="animate-pulse flex space-x-4">
                                     <div class="w-16 h-16 bg-white/20 rounded-lg"></div>
                                     <div class="space-y-2">
                                         <div class="h-4 bg-white/20 rounded w-24"></div>
                                         <div class="h-3 bg-white/20 rounded w-32"></div>
                                     </div>
                                 </div>
                             </div>
                         </div>
                         <div id="weather-summary" class="text-right">
                             <div class="animate-pulse space-y-2">
                                 <div class="h-3 bg-white/20 rounded w-20 ml-auto"></div>
                                 <div class="h-3 bg-white/20 rounded w-16 ml-auto"></div>
                             </div>
                         </div>
                     </div>
                     
                     <div class="text-center text-sm opacity-80">
                         Clique pour voir les détails et prévisions
                     </div>
                 </div>
                 
                 <div id="weather-expanded" class="weather-expanded">
                     <div class="weather-expanded-inner">
                         <div class="mb-6">
                             <h4 class="text-lg font-medium mb-4">Prévisions par heure</h4>
                             <div id="hourly-forecast" class="hourly-forecast">
                                 <div class="animate-pulse flex space-x-3">
                                     <div class="hour-item">
                                         <div class="h-3 bg-white/20 rounded w-8 mx-auto mb-2"></div>
                                         <div class="w-8 h-8 bg-white/20 rounded mx-auto mb-2"></div>
                                         <div class="h-3 bg-white/20 rounded w-8 mx-auto"></div>
                                     </div>
                                     <div class="hour-item">
                                         <div class="h-3 bg-white/20 rounded w-8 mx-auto mb-2"></div>
                                         <div class="w-8 h-8 bg-white/20 rounded mx-auto mb-2"></div>
                                         <div class="h-3 bg-white/20 rounded w-8 mx-auto"></div>
                                     </div>
                                     <div class="hour-item">
                                         <div class="h-3 bg-white/20 rounded w-8 mx-auto mb-2"></div>
                                         <div class="w-8 h-8 bg-white/20 rounded mx-auto mb-2"></div>
                                         <div class="h-3 bg-white/20 rounded w-8 mx-auto"></div>
                                     </div>
                                 </div>
                             </div>
                         </div>
                         
                         <div class="mb-6">
                             <h4 class="text-lg font-medium mb-4">Prévisions 3 prochains jours</h4>
                             <div id="forecast-days">
                                 <div class="animate-pulse space-y-3">
                                     <div class="h-16 bg-white/20 rounded-xl"></div>
                                     <div class="h-16 bg-white/20 rounded-xl"></div>
                                     <div class="h-16 bg-white/20 rounded-xl"></div>
                                 </div>
                             </div>
                         </div>
                         
                         <div class="weather-source">
                             Données météo fournies par L'IRM. La météo n'est pas une science exacte.
                         </div>
                     </div>
                 </div>
             </div>
             
             
             
             <div id="pub-block" class="bg-white dark:bg-stone-900 rounded-2xl md:rounded-3xl overflow-hidden shadow-lg dark:shadow-stone-900/30 cursor-pointer transition-transform hover:scale-105">
                <div class="aspect-square relative">
                    <div class="absolute inset-0 flex flex-col justify-end p-6 bg-gradient-to-t from-black/60 to-transparent">
                        <div class="absolute top-4 right-4 bg-white/10 text-white text-xs px-2 py-1 rounded-full backdrop-blur-sm">
                            <a class="underline" href="https://aktascorp.com/ads">Publicité</a>
                        </div>
                        <div class="max-w-xs space-y-3">
                            <h3 id="pub-title" class="text-white text-xl font-medium leading-tight"></h3>
                            <p id="pub-description" class="text-white/90 text-sm leading-relaxed"></p>
                            <button class="bg-white/15 text-white px-4 py-2 rounded-full backdrop-blur-sm hover:bg-white/25 transition-colors text-sm">
                                Explorer
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            
            <style>
            .pub-image-bg-1 {
                background-image: url('AceclubGstaad_FIlmLookDSCF6120.jpg');
                background-size: cover;
                background-position: center;
            }
            .pub-image-bg-2 {
                background-image: url('https://au.pandora.net/dw/image/v2/BKNF_PRD/on/demandware.static/-/Sites-pandora-master-catalog/default/dwc4157e2a/productimages/modeldetailshot_rect/Q224_C_PDP_Model_Single_03_1x1_RGB_necklace.jpg?sw=1200&sh=1200&sm=fit&sfrm=png&bgcolor=F5F5F5');
                background-size: cover;
                background-position: center;
            }
            </style>
            
            <script>
            document.addEventListener('DOMContentLoaded', function() {
                const pubBlock = document.getElementById('pub-block');
                const pubTitle = document.getElementById('pub-title');
                const pubDescription = document.getElementById('pub-description');
                
                const ads = [
                    {
                        url: "https://theaceclubparis.com/en-be",
                        bgClass: "pub-image-bg-1",
                        title: "The Swiss Open Gstaad '25",
                        description: "Découvrez la nouvelle collection du Ace Club"
                    },
                    {
                        url: "https://be.pandora.net",
                        bgClass: "pub-image-bg-2", 
                        title: "Votre élégance, signée Pandora",
                        description: "Laissez-vous séduire par des bijoux qui racontent votre histoire. Chaque création Pandora est une invitation à l'exploration."
                    }
                ];
                
                const randomAd = ads[Math.floor(Math.random() * ads.length)];
                
                pubBlock.onclick = () => window.open(randomAd.url, '_blank');
                pubBlock.classList.add(randomAd.bgClass);
                pubTitle.textContent = randomAd.title;
                pubDescription.textContent = randomAd.description;
            });
            </script>
            
            
<?php
if (!$is_guest && $resident['is_blocked'] == 1) {
    $block_reason = htmlspecialchars($resident['reason']);
?>
<style>
.blocked-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.97);
    z-index: 20000;
    display: flex;
    align-items: center;
    justify-content: center;
    backdrop-filter: blur(20px);
}

.blocked-content {
    max-width: 500px;
    width: 90%;
    padding: 48px 40px;
    background: white;
    border-radius: 24px;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    text-align: center;
    position: relative;
}

.dark .blocked-content {
    background: black;
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.blocked-icon {
    width: 80px;
    height: 80px;
    background: linear-gradient(135deg, #dc2626 0%, #991b1b 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 24px;
}

.blocked-title {
    font-size: 28px;
    font-weight: 300;
    color: #000;
    margin-bottom: 12px;
    letter-spacing: -0.02em;
}

.dark .blocked-title {
    color: #fff;
}

.blocked-subtitle {
    font-size: 16px;
    color: #666;
    margin-bottom: 32px;
    font-weight: 400;
}

.dark .blocked-subtitle {
    color: #999;
}

.blocked-reason-box {
    background: rgba(220, 38, 38, 0.1);
    border: 1px solid rgba(220, 38, 38, 0.3);
    border-radius: 16px;
    padding: 24px;
    margin-bottom: 32px;
}

.dark .blocked-reason-box {
    background: rgba(220, 38, 38, 0.15);
}

.blocked-reason-label {
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    color: #dc2626;
    margin-bottom: 8px;
}

.dark .blocked-reason-label {
    color: #f87171;
}

.blocked-reason-text {
    font-size: 15px;
    color: #991b1b;
    line-height: 1.6;
}

.dark .blocked-reason-text {
    color: #fca5a5;
}

.blocked-actions {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.blocked-btn-primary {
    width: 100%;
    padding: 16px;
    background: #dc2626;
    border: none;
    border-radius: 12px;
    color: white;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    text-decoration: none;
    display: inline-block;
}

.blocked-btn-primary:hover {
    background: #b91c1c;
    transform: translateY(-2px);
}

.blocked-btn-secondary {
    width: 100%;
    padding: 16px;
    background: rgba(0, 0, 0, 0.05);
    border: none;
    border-radius: 12px;
    color: #000;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    text-decoration: none;
    display: inline-block;
}

.dark .blocked-btn-secondary {
    background: rgba(255, 255, 255, 0.1);
    color: #fff;
}

.blocked-btn-secondary:hover {
    background: rgba(0, 0, 0, 0.1);
}

.dark .blocked-btn-secondary:hover {
    background: rgba(255, 255, 255, 0.15);
}

@media (max-width: 640px) {
    .blocked-content {
        padding: 40px 24px;
    }
    
    .blocked-title {
        font-size: 24px;
    }
    
    .blocked-subtitle {
        font-size: 14px;
    }
    
    .blocked-icon {
        width: 64px;
        height: 64px;
    }
}
</style>

<div class="blocked-overlay">
    <div class="blocked-content">
        <div class="blocked-icon">
            <svg class="w-10 h-10 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
            </svg>
        </div>
        
        <h1 class="blocked-title">Compte suspendu</h1>
        <p class="blocked-subtitle">Ton accès à Vert Chasseur a été temporairement restreint</p>
        
        <div class="blocked-reason-box">
            <div class="blocked-reason-label">Raison de la suspension</div>
            <div class="blocked-reason-text"><?= $block_reason ?></div>
        </div>
        
        <div class="blocked-actions">
            <a href="https://www.imators.com/support" target="_blank" class="blocked-btn-primary">
                Contacter le support
            </a>
            <a href="/logout" class="blocked-btn-secondary">
                Se déconnecter
            </a>
        </div>
    </div>
</div>

<script>
document.body.style.overflow = 'hidden';
</script>
<?php
}
?>
            
        <div class="bg-white dark:bg-stone-900 rounded-2xl md:rounded-3xl p-6 md:p-8 border border-stone-200 dark:border-stone-800 shadow-lg dark:shadow-stone-900/30">
            <h2 class="text-2xl md:text-3xl font-light text-stone-800 dark:text-white mb-6 md:mb-8 tracking-tight">Le poème de la semaine</h2>
            <p class="text-sm text-stone-500 dark:text-stone-400 mt-6 text-left italic">Par Alois Passareiro</p>
            
            <div id="poem-container" class="relative">
                <div id="poem-content" class="mt-8 text-stone-700 dark:text-stone-300 leading-relaxed space-y-4 transition-all duration-500 overflow-hidden" style="max-height: 200px;">
                    <p class="italic font-semibold text-center mb-6">Les bois</p>
                    <p class="italic">Je rêve avec tristesse aux joyeux louvoiements,</p>
                    <p class="italic">Au gré de la futaie aux ombrages paisibles,</p>
                    <p class="italic">Dont sans fin se profile un dédale charmant,</p>
                    <p class="italic">Où divaguent à loisir des sentiers insensibles…</p>
                    <p class="italic mt-6">Puisse les heures seules et l'angélus surprendre</p>
                    <p class="italic">Tous ces désœuvrements de rêveries jonchés ;</p>
                    <p class="italic">Puisse un carillon seul à ses bois m'arracher ;</p>
                    <p class="italic">Puisse derrière moi les ténèbres s'épandre.</p>
                </div>
                
                <div id="poem-gradient" class="absolute bottom-0 left-0 right-0 h-16 bg-gradient-to-t from-white dark:from-stone-900 to-transparent pointer-events-none transition-opacity duration-500"></div>
                
                <button id="poem-toggle" onclick="togglePoem()" class="mt-4 text-green-600 dark:text-green-400 hover:text-green-700 dark:hover:text-green-300 font-medium text-sm transition-colors flex items-center gap-2">
                    <span id="poem-toggle-text">Voir plus</span>
                    <svg id="poem-toggle-icon" class="w-4 h-4 transition-transform duration-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7"/>
                    </svg>
                </button>
            </div>
        </div>

             <div id="offers-section" class="bg-white dark:bg-stone-900 rounded-2xl md:rounded-3xl p-6 md:p-8 border border-stone-200 dark:border-stone-800 shadow-lg dark:shadow-stone-900/30">
                 <h2 class="text-2xl md:text-3xl font-light text-stone-800 dark:text-white mb-6 md:mb-8 tracking-tight">Offres du moment</h2>
                 
                 <?php if (mysqli_num_rows($offers) > 0 && $is_member): ?>
                     <div class="grid gap-4 md:gap-6">
                         <?php while($offer = mysqli_fetch_assoc($offers)): ?>
                         <div class="bg-gradient-to-r from-green-50 to-emerald-50 dark:from-green-900/20 dark:to-emerald-900/20 p-4 md:p-6 rounded-xl md:rounded-2xl border border-green-100 dark:border-green-800">
                             <div class="flex flex-col sm:flex-row sm:items-start justify-between space-y-3 sm:space-y-0">
                                 <div class="flex-1">
                                     <h3 class="text-base md:text-lg font-medium text-stone-900 dark:text-white mb-1"><?= htmlspecialchars($offer['shop_name']) ?></h3>
                                     <p class="text-green-700 dark:text-green-400 font-medium mb-2 text-sm md:text-base"><?= htmlspecialchars($offer['offer_title']) ?></p>                                    
                                     <?php if ($offer['time_restriction']): ?>
                                     <p class="text-stone-600 dark:text-stone-300 text-xs md:text-sm">Valable <?= htmlspecialchars($offer['time_restriction']) ?></p>
                                     <?php endif; ?>
                                     <?php if ($offer['conditions']): ?>
                                     <p class="text-stone-500 dark:text-stone-400 text-xs mt-1"><?= htmlspecialchars($offer['conditions']) ?></p>
                                     <?php endif; ?>
                                 </div>
                                 <div class="flex-shrink-0">
                                     <span class="bg-green-100 dark:bg-green-800 text-green-800 dark:text-green-200 px-3 py-1 rounded-full text-xs font-medium">
                                         Jusqu'au <?= date('d/m', strtotime($offer['valid_until'])) ?>
                                     </span>
                                 </div>
                             </div>
                         </div>
                         <?php endwhile; ?>
                     </div>
                 <?php else: ?>
                     <div class="text-center py-8 md:py-12">
                         <svg class="w-12 h-12 md:w-16 md:h-16 text-stone-300 dark:text-stone-600 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                             <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v13m0-13V6a2 2 0 112 2h-2zm0 0V5.5A2.5 2.5 0 109.5 8H12zm-7 4h14M5 12a2 2 0 110-4h14a2 2 0 110 4M5 12v7a2 2 0 002 2h10a2 2 0 002-2v-7"/>
                         </svg>
                         <p class="text-stone-500 dark:text-stone-400"><?= $is_member ? 'Aucune offre disponible pour le moment' : 'Crée un compte pour voir les offres' ?></p>
                         <p class="text-stone-400 dark:text-stone-500 text-sm mt-1"><?= $is_member ? 'Reviens bientôt pour découvrir de nouvelles promotions !' : 'Accède gratuitement aux réductions de nos partenaires' ?></p>
                     </div>
                 <?php endif; ?>
             </div>
         </div>

         <div class="space-y-6 md:space-y-8">
             <?php if ($is_guest): ?>
             <div class="bg-gradient-to-br from-amber-50 to-orange-50 dark:from-amber-900/20 dark:to-orange-900/20 rounded-2xl md:rounded-3xl p-6 md:p-8 border border-amber-200 dark:border-amber-800 shadow-lg">
                 <div class="flex items-center mb-4">
                     <svg class="w-6 h-6 text-amber-600 dark:text-amber-400 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                         <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                     </svg>
                     <h3 class="text-xl md:text-2xl font-light text-amber-800 dark:text-amber-200 tracking-tight">Mode Invité</h3>
                 </div>
                 
                 <p class="text-amber-700 dark:text-amber-300 text-sm md:text-base mb-6">Tu navigues en mode invité. Crée un compte pour accéder à toutes les fonctionnalités !</p>
                 
                 <div class="space-y-3 text-xs md:text-sm text-amber-600 dark:text-amber-400 mb-6">
                     <div class="flex items-center space-x-2">
                         <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                             <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                         </svg>
                         <span>Consulter la météo locale</span>
                     </div>
                     <div class="flex items-center space-x-2 opacity-60">
                         <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                             <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                         </svg>
                         <span>Réductions chez les partenaires (nécessite un compte)</span>
                     </div>
                     <div class="flex items-center space-x-2 opacity-60">
                         <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                             <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                         </svg>
                         <span>Carte de membre gratuite (nécessite un compte)</span>
                     </div>
                     <div class="flex items-center space-x-2 opacity-60">
                         <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                             <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                         </svg>
                         <span>Services gratuits du quartier</span>
                     </div>
                 </div>
                 
                 <a href="/login.php" class="block w-full bg-amber-600 hover:bg-amber-700 text-white font-medium py-3 px-4 rounded-lg transition-colors text-center">
                     Créer mon compte
                 </a>
             </div>
             <?php else: ?>
             <div class="bg-white dark:bg-stone-900 rounded-2xl md:rounded-3xl p-6 md:p-8 border border-stone-200 dark:border-stone-800 shadow-lg dark:shadow-stone-900/30">
                 <h3 class="text-xl md:text-2xl font-light text-stone-800 dark:text-white mb-4 md:mb-6 tracking-tight">Mes informations</h3>
                 
                 <div class="space-y-4 md:space-y-6">
                     <div>
                         <label class="text-xs md:text-sm font-medium text-stone-700 dark:text-stone-300 block mb-1">Nom complet</label>
                         <p class="text-sm md:text-base text-stone-900 dark:text-white"><?= htmlspecialchars($resident['first_name'] . ' ' . $resident['last_name']) ?></p>
                     </div>
                     
                     <div>
                         <label class="text-xs md:text-sm font-medium text-stone-700 dark:text-stone-300 block mb-1">Email</label>
                         <p class="text-sm md:text-base text-stone-900 dark:text-white break-all"><?= htmlspecialchars($resident['email']) ?></p>
                     </div>
                     
                     <div>
                         <label class="text-xs md:text-sm font-medium text-stone-700 dark:text-stone-300 block mb-1">Adresse</label>
                         <p class="text-sm md:text-base text-stone-900 dark:text-white"><?= htmlspecialchars($resident['address']) ?></p>
                     </div>
                     
                     <div>
                         <label class="text-xs md:text-sm font-medium text-stone-700 dark:text-stone-300 block mb-1">Statut</label>
                         <p class="text-sm md:text-base text-green-600 dark:text-green-400 font-medium">
                             <?= $is_premium ? 'Membre Premium' : 'Membre Standard' ?>
                         </p>
                     </div>
                     
                     <div>
                         <label class="text-xs md:text-sm font-medium text-stone-700 dark:text-stone-300 block mb-1">Réductions</label>
                         <p class="text-sm md:text-base text-stone-900 dark:text-white"><?= $is_premium ? '10% chez les partenaires' : 'Abonne-toi pour débloquer' ?></p>
                     </div>
                     
                     <div>
                         <label class="text-xs md:text-sm font-medium text-stone-700 dark:text-stone-300 block mb-1">Membre depuis</label>
                         <p class="text-sm md:text-base text-stone-900 dark:text-white"><?= date('F Y', strtotime($resident['created_at'])) ?></p>
                     </div>
                 </div>
             </div>
             <?php endif; ?>
         </div>
     </div>
 </div>
</main>

<script>
document.addEventListener('DOMContentLoaded', function() {
    <?php if (!$is_guest): ?>
    document.getElementById('greeting-text').textContent = '<?= htmlspecialchars($resident['first_name']) ?>';
    <?php endif; ?>
    
    loadWeather();
    
    weatherRefreshInterval = setInterval(() => {
        if (document.getElementById('weather-section')) {
            loadWeather();
        }
    }, 10000);
});
</script>


</body>
</html>